import enum

from django.db.models import Q

from cars.core.util import datetime_helper

import cars.settings

from cars.request_aggregator.models.internal_cc_stats import (
    OutgoingCallCenterEntry, CallCenterAltayOutgoingEntry, InternalCCAltayStatus
)
from cars.request_aggregator.models.call_tags import RequestOriginType
from cars.request_aggregator.serializers.common import AggregatedResponseEntry

from ..base.processing_helper import TelephonyProcessingHelperBase


class OutgoingCallStatus(enum.Enum):
    SERVICED = 'serviced'
    NOT_SERVICED = 'not_serviced'


# deprecated
class OutgoingTelephonyProcessingHelper(TelephonyProcessingHelperBase):
    OUTGOING_QUEUE_NAME = 'outgoing'

    def get_requests(self, request=None, limit=None, **kwargs):
        if self._check_phone_not_matched(**kwargs):
            return []

        filters = self._make_call_filters(**kwargs)

        if not filters and limit is None:
            raise Exception('at least one filter must be provided')

        raw_call_entries = (
            OutgoingCallCenterEntry.objects
            .using(cars.settings.DB_RO_ID)
            .filter(**filters)
            .order_by('-time_enter')
            [:limit]
        )

        entry_tags_mapping = self.tag_description_helper.get_call_tags(
            (e.id for e in raw_call_entries),
            Q(request_origin=RequestOriginType.CARSHARING_OUTGOING.value)
        )

        users_mapping = self._make_call_users_mapping(raw_call_entries, request, kwargs)

        access_mapping = self._make_access_mapping(request)

        call_entries = self._transform_raw_entries(raw_call_entries, entry_tags_mapping, users_mapping, access_mapping)

        return call_entries

    def _check_phone_not_matched(self, user=None, phone_number=None, **kwargs):
        # user phone number is not specified neither explicitly nor present in user info
        return user is not None and phone_number is None

    def _make_call_filters(self, phone_number=None, since=None, until=None, staff_entry_binding=None, **kwargs):
        filters = {}

        if phone_number is not None:
            filters['phone'] = phone_number

        if since is not None:
            filters['time_enter__gte'] = since

        if until is not None:
            filters['time_enter__lt'] = until

        if staff_entry_binding is not None:
            filters['staff_entry_binding'] = staff_entry_binding

        return filters

    def _transform_raw_entries(self, call_entries, entry_tags_mapping, users_mapping, access_mapping):
        response_entries = (
            self._process_call_entry(entry, entry_tags_mapping, users_mapping, access_mapping)
            for entry in call_entries
        )

        response_entries_as_mappings = [e._asdict() for e in response_entries]

        return response_entries_as_mappings

    def _process_call_entry(self, entry, entry_tags_mapping, users_mapping, access_mapping):
        time_enter = entry.time_enter.timestamp()
        time_exit = entry.time_exit.timestamp()

        is_answered = (entry.time_connect is not None)
        status = OutgoingCallStatus.SERVICED if is_answered else OutgoingCallStatus.NOT_SERVICED
        time_connect = entry.time_connect.timestamp() if is_answered else None

        duration = entry.duration
        duration_print = datetime_helper.duration_to_str(duration)

        phone = str(entry.phone) if entry.phone is not None else entry.phone

        agent_info_entry = self.staff_info_helper.get_agent_entry(work_phone=entry.agent)
        agent = self.staff_info_helper.format_staff_entry(agent_info_entry)

        if access_mapping['has_access_to_operators']:
            operators = [agent]
        else:
            operators = []

        tags = entry_tags_mapping.get(str(entry.id), [])

        data_url = None
        data = {'track_url': data_url}

        request_description = {
            'id': entry.id,
            'origin': RequestOriginType.CARSHARING_OUTGOING.name,
            'queue': self.OUTGOING_QUEUE_NAME,
            'type': self.REQUEST_TYPE.value,
        }

        user = users_mapping.get(entry.id, None)

        response_entry = AggregatedResponseEntry(
            time_enter=time_enter, time_exit=time_exit,
            duration=duration, duration_print=duration_print,
            connect_trial_count=1, time_connect=time_connect, status=status.value,
            phone=phone, operators=operators, tags=tags,
            request=request_description, user=user,
            data_url=data_url, data=data, message=None,
        )
        return response_entry


class CCInternalAltayOutProcessingHelper(TelephonyProcessingHelperBase):
    OUTGOING_QUEUE_NAME = 'outgoing'

    DATA_URL_TEMPLATE = (
        'https://altay-callcenter.common-ext.yandex-team.ru/v1/callcenter/call/'
        '{call_id}/recording?user={user_uid}&call_id={call_id}'
    )

    def __init__(self, robot_uid, call_track_view_actions, denied_operator_view_roles, denied_data_view_roles):
        super().__init__(call_track_view_actions, denied_operator_view_roles, denied_data_view_roles)
        self._robot_uid = robot_uid

    @classmethod
    def from_settings(cls):
        settings = cars.settings.REQUEST_AGGREGATOR
        robot_uid = settings['altay']['robot_uid']
        call_track_view_actions = settings['callcenter']['call_track_view_actions']
        denied_operator_view_roles = settings['callcenter']['denied_operator_view_roles']
        denied_data_view_roles = settings['callcenter']['denied_data_view_roles']
        return cls(robot_uid, call_track_view_actions, denied_operator_view_roles, denied_data_view_roles)

    def get_call_url(self, call_id, user_uid=None):
        if call_id is not None:
            user_uid = user_uid or self._robot_uid
            return self.DATA_URL_TEMPLATE.format(user_uid=user_uid, call_id=call_id)
        return None

    def get_requests(self, request=None, limit=None, **kwargs):
        if self._check_phone_not_matched(**kwargs):
            return []

        filters = self._make_call_filters(**kwargs)

        if not filters and limit is None:
            raise Exception('at least one filter must be provided')

        raw_call_entries = (
            CallCenterAltayOutgoingEntry.objects
            .using(cars.settings.DB_RO_ID)
            .filter(**filters)
            .order_by('-time_enter')
            [:limit]
        )

        entry_tags_mapping = self.tag_description_helper.get_call_tags(
            (e.id for e in raw_call_entries),
            Q(request_origin=RequestOriginType.CC_INTERNAL_ALTAY_OUTGOING.value)
        )

        users_mapping = self._make_call_users_mapping(raw_call_entries, request, kwargs)

        access_mapping = self._make_access_mapping(request)

        call_entries = self._transform_raw_entries(raw_call_entries, entry_tags_mapping, users_mapping, access_mapping)

        return call_entries

    def _check_phone_not_matched(self, user=None, phone_number=None, **kwargs):
        # user phone number is not specified neither explicitly nor present in user info
        return user is not None and phone_number is None

    def _make_call_filters(self, phone_number=None, since=None, until=None, staff_entry_binding=None, **kwargs):
        filters = {}

        if phone_number is not None:
            filters['phone'] = phone_number

        if since is not None:
            filters['time_enter__gte'] = since

        if until is not None:
            filters['time_enter__lt'] = until

        if staff_entry_binding is not None:
            filters['staff_entry_binding'] = staff_entry_binding

        return filters

    def _transform_raw_entries(self, call_entries, entry_tags_mapping, users_mapping, access_mapping):
        response_entries = (
            self._process_call_entry(entry, entry_tags_mapping, users_mapping, access_mapping)
            for entry in call_entries
        )

        response_entries_as_mappings = [e._asdict() for e in response_entries]

        return response_entries_as_mappings

    def _process_call_entry(self, entry, entry_tags_mapping, users_mapping, access_mapping):
        time_enter = entry.time_enter.timestamp()
        time_connect = entry.time_connect.timestamp() if entry.time_connect is not None else None
        time_exit = entry.time_exit.timestamp()

        if entry.status in (InternalCCAltayStatus.SUCCESS.value, InternalCCAltayStatus.ANSWER_PHONE.value):
            status = OutgoingCallStatus.SERVICED
        else:
            status = OutgoingCallStatus.NOT_SERVICED

        duration = entry.duration
        duration_print = datetime_helper.duration_to_str(duration)

        phone = str(entry.phone) if entry.phone is not None else entry.phone

        agent_info_entry = self.staff_info_helper.get_agent_entry(id=entry.staff_entry_binding_id)
        agent = self.staff_info_helper.format_staff_entry(agent_info_entry)

        if access_mapping['has_access_to_operators']:
            operators = [agent]
        else:
            operators = []

        tags = entry_tags_mapping.get(str(entry.id), [])

        if access_mapping['has_access_to_call_id']:
            user_uid = None  # to use request user: getattr(request.user, 'uid', None) if request is not None else None
            data_url = self.get_call_url(entry.external_id, user_uid) if entry.has_record else None
        else:
            data_url = None

        data = {
            'track_url': data_url,
            'comment': entry.comment,
        }

        request_description = {
            'id': entry.id,
            'origin': RequestOriginType.CC_INTERNAL_ALTAY_OUTGOING.name,
            'queue': self.OUTGOING_QUEUE_NAME,
            'type': self.REQUEST_TYPE.value,
        }

        user = users_mapping.get(entry.id, None)

        response_entry = AggregatedResponseEntry(
            time_enter=time_enter, time_exit=time_exit,
            duration=duration, duration_print=duration_print,
            connect_trial_count=1, time_connect=time_connect, status=status.value,
            phone=phone, operators=operators, tags=tags,
            request=request_description, user=user,
            data=data, message=None, data_url=data_url,
        )
        return response_entry
