import logging

from django.db import transaction

from cars.core.util import phone_number_helper, datetime_helper
from cars.callcenter.core import StaffInfoHelper

from cars.request_aggregator.core.phone_binding_helper import PhoneBindingHelper
from cars.request_aggregator.models.internal_cc_stats import OutgoingCallCenterEntry
from cars.request_aggregator.models.call_center_common import CallStatSyncStatus, SyncOrigin


LOGGER = logging.getLogger(__name__)


class OutgoingTelephonyServiceHelper(object):
    MAX_ENTRIES_LIMIT = 1000

    phone_binding_helper = PhoneBindingHelper()
    staff_info_helper = StaffInfoHelper.make_default()

    @classmethod
    def from_settings(cls):
        return cls()

    def update_outgoing_bad_formatted_phone_numbers(self):
        with transaction.atomic(savepoint=False):
            bad_phone_number_entries = (
                OutgoingCallCenterEntry.objects.select_for_update()
                .filter(phone__startswith='+98')
                [:self.MAX_ENTRIES_LIMIT]
            )

            for entry in bad_phone_number_entries:
                origin_phone = str(entry.phone).lstrip('+')
                normalized_phone = phone_number_helper.normalize_phone_number(origin_phone)

                LOGGER.info('update number "{}" to "{}"'.format(origin_phone, normalized_phone))

                entry.phone = normalized_phone
                entry.save()

        total_numbers = len(bad_phone_number_entries)
        LOGGER.info('total numbers updated: {}'.format(total_numbers))

        return total_numbers

    def update_outgoing_staff_bindings(self, offset, count):
        limit = min(count, self.MAX_ENTRIES_LIMIT)

        with transaction.atomic(savepoint=False):
            entries = (
                OutgoingCallCenterEntry.objects.select_for_update()
                .filter(
                    staff_entry_binding__isnull=True,
                )
                .order_by('id')
                [offset:offset + limit]
            )

            entries = list(entries)

            for entry in entries:
                agent_entry = self.staff_info_helper.get_agent_entry(work_phone=entry.agent)
                if agent_entry is not None:
                    entry.staff_entry_binding = agent_entry
                    entry.save()

            last_entry_id = entries[-1].id if entries else None
            values_to_update = {
                'last_data_sync_time': datetime_helper.utc_now(),
            }

            if last_entry_id is not None:
                values_to_update['data'] = {'last_entry_id': last_entry_id}

            sync_entry = CallStatSyncStatus.objects.filter(
                origin=SyncOrigin.SERVICE_CC_INTERNAL_OUT_UPDATE_STAFF_BINDING.value,
            ).first()

            if sync_entry is None:
                sync_entry = CallStatSyncStatus.objects.create(
                    origin=SyncOrigin.SERVICE_CC_INTERNAL_OUT_UPDATE_STAFF_BINDING.value, **values_to_update
                )
            else:
                for key, value in values_to_update.items():
                    setattr(sync_entry, key, value)
                sync_entry.save()

        total_numbers = len(entries)
        LOGGER.info('total numbers updated: {}'.format(total_numbers))

        return total_numbers
