from collections import namedtuple
import datetime
import itertools
import logging

from cars.core.util import datetime_helper

from cars.request_aggregator.core.request_time_sync_helper import RequestTimeSyncHelper, RequestStateSyncHelper
from cars.request_aggregator.models.call_center_common import SyncOrigin

from .api_helper import AltayApiHelper

LOGGER = logging.getLogger(__name__)


class OutgoingTelephonyRequestTimeSyncHelper(RequestTimeSyncHelper):
    @classmethod
    def from_settings(cls):
        # collect 45-minutes time range each time, but not more than 6 hours if error occurred
        return cls(
            SyncOrigin.CC_INTERNAL_OUTGOING_ENTRIES,
            default_since=datetime_helper.utc_localize(datetime.datetime(2018, 2, 14)),
            max_time_span=datetime.timedelta(hours=6),
            request_lag=datetime.timedelta(minutes=45),
            request_overlap=datetime.timedelta(seconds=0),
        )


class CCInternalAltayOutSyncHelper(RequestStateSyncHelper):
    State = namedtuple('State', ('uid', 'offset', 'limit'))

    LIMIT_STEPS = (AltayApiHelper.DEFAULT_LIMIT, 100, 1000, AltayApiHelper.MAX_LIMIT)
    MIN_LIMIT, MAX_LIMIT = LIMIT_STEPS[0], LIMIT_STEPS[-1]

    def __init__(self, stat_sync_origin, *, default_state=None):
        super().__init__(stat_sync_origin, default_state=default_state)
        self._extra_states = []

    @classmethod
    def from_settings(cls):
        return cls(
            SyncOrigin.CC_INTERNAL_ALTAY_OUTGOING_ENTRIES,
            default_state={'offset': 0, 'limit': cls.MIN_LIMIT, 'extra_states': []}
        )

    def _get_state(self, sync_entry):
        return self.State(uid=None, offset=sync_entry.data['offset'], limit=sync_entry.data['limit'])

    def _update_state(self, sync_entry, state=None):
        # state won't change
        sync_entry.data['extra_states'].extend(self._extra_states)
        self._extra_states.clear()

        # perform only one general operation
        has_more = False
        return has_more

    def add_extra_state(self, uid, *, current_offset, current_limit):
        assert current_limit <= self.MAX_LIMIT

        offset = current_offset + current_limit
        limit = next(itertools.dropwhile(lambda x: x <= current_limit, self.LIMIT_STEPS), self.MAX_LIMIT)

        state = self.State(uid=uid, offset=offset, limit=limit)

        LOGGER.info('sync origin {}: add extra state {}'.format(self.sync_origin, state))

        self._extra_states.append(state)

    def _iter_extra_states(self, sync_entry):
        extra_states = sync_entry.data['extra_states']

        for _ in range(len(extra_states)):  # do not process newly added states
            state = extra_states.pop(0)

            LOGGER.info('sync origin {}: processing extra state {}'.format(self.sync_origin, state))

            yield state

            extra_states.extend(self._extra_states)
            self._extra_states.clear()

            sync_entry.save()
