import collections
import datetime
import logging

from cars.core.telephony import IncomingTelephonyApiHelper as IncomingTelephonyApiHelperBase
from cars.core.util import datetime_helper
from cars.request_aggregator.models.call_center_common import RoutingVerb


LOGGER = logging.getLogger(__name__)


class RegisterCallApiHelper(object):
    CALL_INCOMING_API_RESPONSE = collections.namedtuple(
        'CALL_INCOMING_API_RESPONSE',
        ['time_id', 'call_id', 'verb', 'source_cc', 'phone', ]
    )

    CALL_ROUTING_API_RESPONSE = collections.namedtuple(
        'CALL_ROUTING_API_RESPONSE',
        ['time_id', 'call_id', 'verb', 'source_cc', 'target_cc', 'phone', 'fallback', ]
    )

    CALL_ASSIGNMENT_API_RESPONSE = collections.namedtuple(
        'CALL_ASSIGNMENT_API_RESPONSE',
        IncomingTelephonyApiHelperBase.API_RESPONSE._fields
    )

    @classmethod
    def from_settings(cls):
        return cls()

    def process_data(self, entries):
        call_income_entries = []
        call_routing_entries = []
        call_assignment_entries = []

        for entry in entries:
            if entry['verb'] == RoutingVerb.INCOMING.value:
                processed_entry = self._process_call_incoming_entry(entry)
                call_income_entries.append(processed_entry)
            elif entry['verb'] == RoutingVerb.DISTRIBUTE.value:
                processed_entry = self._process_call_routing_entry(entry)
                call_routing_entries.append(processed_entry)
            elif entry['verb'] == RoutingVerb.XCONNECT.value:
                processed_entry = self._process_call_assignment_entry(entry)
                call_assignment_entries.append(processed_entry)
            else:
                LOGGER.error('incorrect partition or unknown verb in entry: "{}"'.format(entry))
                raise Exception('incorrect partition or unknown verb')

        return call_income_entries, call_routing_entries, call_assignment_entries

    def _process_call_incoming_entry(self, entry):
        return self.CALL_INCOMING_API_RESPONSE(
            time_id=datetime_helper.timestamp_to_datetime(entry['time_id']),
            call_id=entry['call_id'],
            verb=entry['verb'],
            source_cc=entry['data1'],
            phone=entry['data3'],
        )

    def _process_call_routing_entry(self, entry):
        return self.CALL_ROUTING_API_RESPONSE(
            time_id=datetime_helper.timestamp_to_datetime(entry['time_id']),
            call_id=entry['call_id'],
            verb=entry['verb'],
            source_cc=entry['data1'],
            target_cc=entry['data2'],
            phone=entry['data3'],
            fallback=entry['data4'],
        )

    def _process_call_assignment_entry(self, entry):
        try:
            user_waiting_time = int(entry['data1'])
        except (ValueError, TypeError):
            user_waiting_time = 0

        user_waiting_time = datetime.timedelta(seconds=user_waiting_time)

        return self.CALL_ASSIGNMENT_API_RESPONSE(
            time_id=datetime_helper.timestamp_to_datetime(entry['time_id']),
            queue=entry['queue'],
            call_id=entry['call_id'],
            agent=entry['agent'],
            verb=entry['verb'],
            data=user_waiting_time,
        )
