import datetime
import logging

from cars.core.telephony import TelephonyQueue, DYNAMIC_TELEPHONY_QUEUE
from cars.core.util import datetime_helper

from cars.request_aggregator.core.common_helper import collection_to_mapping
from cars.request_aggregator.core.request_time_sync_helper import RequestTimeSyncHelper
from cars.request_aggregator.core.statistics import StatisticsHelperBase, register

from cars.request_aggregator.models.internal_cc_stats import CallCenterEntry, InternalCCVerb, InternalCCVerbPrecedence
from cars.request_aggregator.models.call_center_common import SyncOrigin

LOGGER = logging.getLogger(__name__)


@register('internal_cc')
class InternalCCStatisticsHelper(StatisticsHelperBase):
    def __init__(self):
        super().__init__()
        self._entries_sync_helper = RequestTimeSyncHelper(
            use_timestamp=False,
            stat_sync_origin=SyncOrigin.CC_INTERNAL_STAT_MONITORING,
            default_since=datetime_helper.utc_now(),
            request_lag=datetime.timedelta(minutes=1),
            max_time_span=datetime.timedelta(minutes=1),
        )
        self._operators_sync_helper = RequestTimeSyncHelper(
            use_timestamp=False,
            stat_sync_origin=SyncOrigin.CC_INTERNAL_OPERATOR_MONITORING,
            default_since=datetime_helper.utc_now(),
            request_lag=datetime.timedelta(hours=12),
            max_time_span=datetime.timedelta(hours=12),
        )

    def collect(self):
        stats = {}

        for since, until in self._entries_sync_helper.iter_time_span_to_process():
            self.update_collection(stats, self.get_call_entries_stat(since, until))

        for since, until in self._operators_sync_helper.iter_time_span_to_process():
            self.update_collection(stats, self.get_operators_stat(since, until))

        return stats

    def get_operator_activity(self, since, until, *, select_related=True):
        operator_activity_entries = (
            CallCenterEntry.objects
            .filter(
                verb__in=(InternalCCVerb.ADD_MEMBER.value, InternalCCVerb.REMOVE_MEMBER.value),
                time_id__gte=since,
                time_id__lt=until,
            )
            .order_by('time_id')
        )

        if select_related:
            operator_activity_entries = operator_activity_entries.select_related('staff_entry_binding')

        operator_activity_mapping = collection_to_mapping(
            operator_activity_entries, attr_key='agent', multiple_values=False
        )
        return operator_activity_mapping

    def get_active_operators(self, since, until):
        operator_activity = self.get_operator_activity(since, until)
        active_operators = [
            v.staff_entry_binding for v in operator_activity.values()
            if v.verb == InternalCCVerb.ADD_MEMBER.value
        ]
        return active_operators

    def get_operators_stat(self, since, until):
        return {'active_operators': len(self.get_active_operators(since, until))}

    def get_call_entries_stat(self, since, until):
        queue_stats = {
            queue.value: self.get_default_queue_stat_collection()
            for queue in TelephonyQueue
        }
        queue_stats[DYNAMIC_TELEPHONY_QUEUE] = self.get_default_queue_stat_collection()

        intro_call_queue_names = self._get_intro_call_queue_names(since, until)
        for queue_name in intro_call_queue_names:
            queue_stats[queue_name]['total'] += 1

        call_ids = self._get_call_ids(since, until)
        call_entries_mapping = self._compile_call_entries(call_ids)

        for call_id, call_parts in call_entries_mapping.items():
            stats, call_parts = self._prepare_call_entry(queue_stats, call_parts)
            self._update_call_entry_parts_stat(stats, call_parts)

        self.annotate_call_duration(queue_stats)

        return queue_stats

    def _get_intro_call_queue_names(self, since, until):
        call_queue_names = (
            CallCenterEntry.objects
            .filter(verb=InternalCCVerb.ENTER_QUEUE.value, time_id__gte=since, time_id__lt=until)
            .values_list('queue_name', flat=True)
        )
        return call_queue_names

    def _get_call_ids(self, since, until):
        call_entries = (
            CallCenterEntry.objects
            .filter(time_id__gte=since, time_id__lt=until)
            .values_list('call_id', flat=True)
        )
        return [x for x in call_entries if x != 'REALTIME']

    def _compile_call_entries(self, call_ids):
        all_entries = (
            CallCenterEntry.objects.filter(call_id__in=call_ids)
            .order_by('time_id')
            .values('call_id', 'verb', 'time_id', 'queue_name')
        )
        all_entries_mapping = collection_to_mapping(all_entries, item_key='call_id')
        return all_entries_mapping

    def _prepare_call_entry(self, queue_stats, call_parts):
        queue_name = call_parts[0]['queue_name']
        stats = queue_stats[queue_name]

        call_parts.sort(key=lambda x: InternalCCVerbPrecedence.get(InternalCCVerb.try_make(x['verb'])))

        return stats, call_parts

    def _update_call_entry_parts_stat(self, stats, entry_parts):
        last_part_verb = InternalCCVerb.try_make(entry_parts[-1]['verb'])

        calculate_duration = False

        if last_part_verb.is_completion_entry() or last_part_verb.is_transfer_entry():
            stats['completed'] += 1
            stats['total_managed'] += 1
            calculate_duration = True

        elif last_part_verb.is_abandon_entry():
            stats['not_serviced'] += 1
            stats['total_managed'] += 1
            calculate_duration = True

        elif last_part_verb.is_connect_entry():
            stats['serviced'] += 1

        else:
            stats['pending'] += 1

        if calculate_duration:
            duration = (entry_parts[-1]['time_id'] - entry_parts[0]['time_id']).total_seconds()
            stats['durations'].append(duration)

        return stats
