import enum
import logging
import time

import requests

import cars.settings

from cars.core.solomon import SolomonHelper
from cars.callcenter.models.call_assignment import CallAssignment, CallAssignmentCallCenter


LOGGER = logging.getLogger(__name__)


class BaseTagReporterMixin(object):
    REPORT_ATTEMPTS_COUNT = 3
    REPORT_SLEEP_TIME_S = 0.2
    REPORT_TIMEOUT_S = 5

    solomon_helper = SolomonHelper('request_aggregator', 'call_assignment_tag_reporter')

    class ReportCallCenter(enum.Enum):
        Audiotele = 'audiotele'
        InternalCC = 'yandex_internal'
        NextContact = 'nextcontact'
        Unknown = 'unknown'

    def _upsert_call_assignment_entries_in_tags(self, entries, *, extra_mark=None):
        return self.__apply_call_assignment_entries_in_tags(entries, to_remove=False, extra_mark=extra_mark)

    def _remove_call_assignment_entries_in_tags(self, entries, *, extra_mark=None):
        return self.__apply_call_assignment_entries_in_tags(entries, to_remove=True, extra_mark=extra_mark)

    def __apply_call_assignment_entries_in_tags(self, entries, *, to_remove, extra_mark=None):
        try:
            payloads = [
                self.__make_payload_from_call_assignment_entry(entry, to_remove=to_remove)
                for entry in entries
            ]
            for payload in payloads:
                self.__report_event(payload, extra_mark=extra_mark)

        except Exception:
            LOGGER.exception('reporting events list failed')

    def __report_event(self, data, *, extra_mark=None):
        _call_data = str(data)

        LOGGER.info('now reporting ' + _call_data)

        for attempt_no in range(self.REPORT_ATTEMPTS_COUNT):
            try:
                r = requests.post(
                    cars.settings.CALLCENTER['call_events_url'],
                    json=data,
                    headers={
                        'Authorization': 'OAuth ' + cars.settings.CALLCENTER['saas_history_token']
                    },
                    timeout=self.REPORT_TIMEOUT_S
                )
                r.raise_for_status()
            except Exception:
                self.__report_solomon_event('attempt_fail', data, extra_mark)
                LOGGER.exception('unable to report call: ' + _call_data)
                if attempt_no < self.REPORT_ATTEMPTS_COUNT - 1:  # not last
                    time.sleep(self.REPORT_SLEEP_TIME_S)
            else:
                break
        else:
            self.__report_solomon_event('report_fail', data, extra_mark)
            LOGGER.error('event reporting failed ' + _call_data)

    def __report_solomon_event(self, event, payload, extra_mark):
        sensor_name_parts = [event, payload.get('cc', 'unknown'), payload.get('type', 'unknown'), extra_mark]
        self.solomon_helper.increment_counter('.'.join(str(x) for x in sensor_name_parts if x))

    def __make_payload_from_call_assignment_entry(self, entry, *, to_remove):
        assert isinstance(entry, CallAssignment), 'entry is not a call assignment'

        payload = {
            'cc': self.__get_report_call_center(entry),
            'queue': self.__get_report_call_queue(entry),
            'phone': str(entry.from_number),
            'call_id': self.__get_report_call_id(entry)
        }

        is_call_assigned = (entry.to_user is not None)

        if is_call_assigned:
            payload['performer'] = str(entry.to_user_id)

        if not to_remove:
            payload['type'] = 'start' if not is_call_assigned else 'set_performer'
        else:
            payload['type'] = 'finish'

        return payload

    def __get_report_call_center(self, entry):
        assert isinstance(entry, CallAssignment)
        if entry.call_center == CallAssignmentCallCenter.Yandex.value:
            return self.ReportCallCenter.InternalCC.value
        elif entry.call_center == CallAssignmentCallCenter.AudioTele.value:
            return self.ReportCallCenter.Audiotele.value
        elif entry.call_center == CallAssignmentCallCenter.NextContact.value:
            return self.ReportCallCenter.NextContact.value
        else:
            return self.ReportCallCenter.Unknown.value

    def __get_report_call_queue(self, entry):
        return entry.queue or ''

    def __get_report_call_id(self, entry):
        return str(entry.call_id)
