import collections
import enum
import json
import logging

from cars.core.util import datetime_helper, phone_number_helper
from cars.callcenter.core import StaffInfoHelper
from cars.users.models import User


LOGGER = logging.getLogger(__name__)


def parse_bool_value(value):
    return str(value).lower() in ('1', 'true')


class NextContactCallDirection(enum.Enum):
    INCOMING = 'incoming'
    OUTGOING = 'outgoing'


class NextContactAPIHelper(object):
    staff_info_helper = StaffInfoHelper.make_default()

    CallEnterEntry = collections.namedtuple(
        'CallEnterEntry', ('direction', 'time_enter', 'time_connect',
                           'phone', 'agent', 'staff_entry_binding', 'call_id', 'meta_info')
    )

    CallExitEntry = collections.namedtuple(
        'CallExitEntry', ('direction', 'time_enter', 'time_connect', 'time_exit', 'duration', 'is_answered',
                          'phone', 'agent', 'staff_entry_binding', 'call_id', 'meta_info')
    )

    def make_call_enter_entry(self, **raw_entry):
        meta_info = {}

        direction = NextContactCallDirection(raw_entry['call_type'])

        call_id = raw_entry['session_id']

        time_enter = time_connect = datetime_helper.now()  # entries are supposed to be real-time

        original_phone = raw_entry['phone_number']
        phone = phone_number_helper.normalize_phone_number(original_phone)

        if phone is None:
            meta_info['original_phone'] = original_phone

        raw_agent = raw_entry['cc_operator_login']
        agent = raw_agent.lower() if raw_agent is not None and raw_agent != 'null' else None
        staff_entry_binding = self.staff_info_helper.get_agent_entry(external_username=agent)

        meta_info['is_answered'] = parse_bool_value(raw_entry['was_answered_by_the_operator'])

        entry = self.CallEnterEntry(
            direction=direction,
            time_enter=time_enter,
            time_connect=time_connect,
            phone=phone,
            agent=agent,
            staff_entry_binding=staff_entry_binding,
            call_id=call_id,
            meta_info=(meta_info or None),
        )
        return entry

    def make_call_exit_entry(self, **raw_entry):
        meta_info = {}

        direction = NextContactCallDirection(raw_entry['call_type'])

        call_id = raw_entry['session_id']

        time_field_names = ('call_arrival_time', 'call_start_time', 'call_finish_time')
        time_enter, time_connect, time_exit = [
            datetime_helper.timestamp_to_datetime(raw_entry[x]) if raw_entry[x] and raw_entry[x] != 'null' else None
            for x in time_field_names
        ]

        if time_exit is None:
            LOGGER.error(
                'entry with id {} has incorrect call finish time: {}'
                .format(raw_entry.get('call_id', None), json.dumps(raw_entry))
            )
            return None

        is_answered = parse_bool_value(raw_entry['was_answered_by_the_operator'])

        duration = int(raw_entry['call_duration']) if is_answered else None

        original_phone = raw_entry['phone_number']
        phone = phone_number_helper.normalize_phone_number(original_phone)

        if phone is None:
            meta_info['original_phone'] = original_phone

        agent = None
        staff_entry_binding = None

        raw_agent = raw_entry['cc_operator_login']
        agent = raw_agent.lower() if raw_agent is not None and raw_agent != 'null' else None
        if agent is not None:
            staff_entry_binding = self.staff_info_helper.get_agent_entry(external_username=agent)

        if staff_entry_binding is None:
            external_username, internal_username, printable_name = (raw_entry['cc_operator_login'],
                                                                    raw_entry['ys_operator_login'],
                                                                    raw_entry['operator_fio'])
            staff_entry_binding = self.staff_info_helper.bind_external_username(
                external_username, internal_username, printable_name
            )

            if staff_entry_binding is not None:
                agent = staff_entry_binding.username
            else:
                LOGGER.error(
                    'Cannot bind NextContact agent: external - {}, internal - {}, printable - {}'
                    .format(external_username, internal_username, printable_name)
                )
                meta_info['original_agent_external_username'] = external_username
                meta_info['original_agent_internal_username'] = internal_username
                meta_info['original_agent_print_name'] = printable_name

        connection_attempts = raw_entry['connection_attempts']
        if connection_attempts > 1:
            meta_info['connection_attempts'] = connection_attempts

        entry = self.CallExitEntry(
            direction=direction, time_enter=time_enter, time_connect=time_connect, time_exit=time_exit,
            duration=duration, is_answered=is_answered,
            phone=phone, agent=agent, staff_entry_binding=staff_entry_binding, call_id=call_id,
            meta_info=(meta_info or None),
        )
        return entry

    def get_generalized_user_data(self, phone, staff_entry):
        from_number = phone_number_helper.normalize_phone_number(phone)

        if from_number:
            # do not add bad formatted entries
            from_user = (
                User.objects.filter(phone=from_number)
                    .order_by('status')  # make "active" first if exists
                    .first()
            )
        else:
            from_user = None

        _to_user_binding = staff_entry

        if _to_user_binding is not None:
            to_user = _to_user_binding.user
            to_number = _to_user_binding.work_phone
        else:
            to_user = None
            to_number = None

        return from_number, to_number, from_user, to_user
