import json
import logging

from cars.callcenter.models.call_assignment import CallAssignment, CallAssignmentCallCenter

from cars.request_aggregator.core.common_helper import atomic_with_retries

from .api_helper import NextContactAPIHelper
from .mixins import NextContactReporterMixin


LOGGER = logging.getLogger(__name__)


class NextContactCallAssignmentHelper(NextContactReporterMixin):
    api_helper = NextContactAPIHelper()

    CC = CallAssignmentCallCenter.NextContact.value
    QUEUE = 'nextcontact-incoming-default'

    def register_call_entries(self, processed_entries):
        entries_to_register = []

        for e in processed_entries:
            time_enter = e.time_enter
            time_connect = e.time_connect
            call_id = e.call_id

            from_number, to_number, from_user, to_user = self.api_helper.get_generalized_user_data(
                e.phone, e.staff_entry_binding
            )

            assignment_entry = CallAssignment(
                from_number=from_number,
                to_number=to_number,
                from_user=from_user,
                to_user=to_user,
                added_at=time_enter,
                connected_at=time_connect,
                call_id=call_id,
                call_center=self.CC,
                queue=self.QUEUE,
            )

            entries_to_register.append(assignment_entry)

            LOGGER.info('call start ({}): call_id - {}, time_enter - {}, phone - {}, agent - {}'
                        .format(self.CC, e.call_id, e.time_enter, e.phone, e.agent))

        self._create_entries_atomic(entries_to_register)
        self._upsert_call_assignment_entries_in_tags(entries_to_register, extra_mark='webhook')

    @atomic_with_retries
    def _create_entries_atomic(self, entries_to_register):
        CallAssignment.objects.bulk_create(entries_to_register)

    def remove_call_entries(self, processed_entries):
        entries_to_remove = []

        for e in processed_entries:
            call_id = e.call_id

            assignment_entries = CallAssignment.objects.filter(
                call_center=CallAssignmentCallCenter.NextContact.value,
                call_id=call_id,
            )
            entries_to_remove.extend(assignment_entries)

            LOGGER.info('call finish ({}): call_id - {}, time_enter - {}, phone - {}, agent - {}, direction - {}'
                        .format(self.CC, e.call_id, e.time_enter, e.phone, e.agent, e.direction))

        self._remove_call_assignment_entries_in_tags(entries_to_remove, extra_mark='webhook')
        self._remove_entries_atomic(entries_to_remove)

    @atomic_with_retries
    def _remove_entries_atomic(self, entries_to_remove):
        for query in entries_to_remove:
            query.delete()


class NextContactWebHookHelper(object):
    api_helper = NextContactAPIHelper()

    assignment_helper = NextContactCallAssignmentHelper()

    @classmethod
    def from_settings(cls):
        return cls()

    def _parse_data(self, data):
        try:
            data = json.loads(data) if isinstance(data, str) else data
        except Exception:
            raise Exception('error parsing data')
        return data

    def process_call_start(self, raw_data):
        data = self._parse_data(raw_data)

        entries = data.get('data', None)

        if entries is None:
            entries = [data]

        if isinstance(entries, list):
            # it's a stub right now
            required_fields = (
                'session_id', 'call_type', 'was_answered_by_the_operator', 'phone_number', 'cc_operator_login'
            )

            if not all(x in e for x in required_fields for e in entries):
                raise Exception('incorrect data format')

            processed_entries = [self.api_helper.make_call_enter_entry(**raw_entry) for raw_entry in entries]

            self.assignment_helper.register_call_entries(processed_entries)
        else:
            raise Exception('incorrect data format: no "data" field in request body')

    def process_call_finish(self, raw_data):
        data = self._parse_data(raw_data)

        entries = data.get('data', None)

        if entries is None:
            entries = [data]

        if isinstance(entries, list):
            # it's a stub right now
            required_fields = (
                'session_id', 'call_type', 'was_answered_by_the_operator', 'phone_number', 'cc_operator_login',
                'call_arrival_time', 'call_start_time', 'call_finish_time', 'call_duration', 'connection_attempts',
                'ys_operator_login', 'operator_fio'
            )

            if not all(x in e for x in required_fields for e in entries):
                raise Exception('incorrect data format')

            processed_entries = []

            for raw_entry in entries:
                entry = self.api_helper.make_call_exit_entry(**raw_entry)
                if entry is not None:
                    processed_entries.append(entry)

            self.assignment_helper.remove_call_entries(processed_entries)
        else:
            raise Exception('incorrect data format: no "data" field in request body')
