from collections import OrderedDict
from itertools import islice

from urllib.parse import urlparse, parse_qs

from rest_framework.pagination import CursorPagination as CursorPaginationBase
from rest_framework.response import Response

from cars.core.util import datetime_helper


class QuerySetWrapMixin(object):
    def __init__(self, helper_instance, get_data_method_name, **kwargs):
        self._helper_instance = helper_instance
        self._get_data_method_name = get_data_method_name

        self._kwargs = None
        self._slice = None

        self.initialize(**kwargs)

    def initialize(self, **kwargs):
        self._kwargs = kwargs
        self._slice = None

    def order_by(self, *ordering):
        # reverse order is not supported now
        assert ordering == ('-time_enter', )
        return self

    def filter(self, **kwargs):
        # reverse order is not supported now
        assert len(kwargs) == 1 and 'time_enter__lt' in kwargs

        modified_upper = datetime_helper.timestamp_to_datetime(kwargs['time_enter__lt'])
        self._kwargs['until'] = modified_upper

        return self

    def __getitem__(self, item):
        assert isinstance(item, slice)
        self._slice = (item.start, item.stop)
        return self

    def __iter__(self):
        assert self._slice is not None
        offset, upper_offset = self._slice
        self._kwargs['limit'] = upper_offset  # overwrite original value as it has been processed by cursor
        results = self._get_results(**self._kwargs)
        return islice(results, offset, upper_offset)

    def _get_results(self, **params):
        get_data = getattr(self._helper_instance, self._get_data_method_name)
        return get_data(**params)


class CursorPagination(CursorPaginationBase):
    page_size = 20
    max_page_size = 50
    page_size_query_param = 'limit'

    cursor_query_param = 'next_cursor'  # reverse order is not supported now

    ordering = ('-time_enter', )

    def get_results(self, data):
        return data['data']

    def get_paginated_response(self, data):
        # reverse order is not supported now
        next_link = self.get_next_link()
        next_cursor = self.get_cursor_from_link(next_link)

        return Response(OrderedDict([
            ('entries_count', len(data)),
            ('data', data),
            ('more', next_link),
            (self.cursor_query_param, next_cursor),
        ]))

    def get_cursor_from_link(self, link):
        cursor = None

        if link is not None:
            parsed_url = urlparse(link)
            url_query = parse_qs(parsed_url.query, keep_blank_values=True)
            cursor_values = url_query.get(self.cursor_query_param, [])
            cursor = next(iter(cursor_values), cursor)

        return cursor


class ChatMessagesCursorPagination(CursorPagination):
    cursor_query_param = 'cursor'  # support two-side order

    ordering = ('-time_id', )

    def get_paginated_response(self, data):
        previous_link = self.get_previous_link()
        previous_cursor = self.get_cursor_from_link(previous_link)

        next_link = self.get_next_link()
        next_cursor = self.get_cursor_from_link(next_link)

        return Response(OrderedDict([
            ('entries_count', len(data)),
            ('data', data),
            ('previous', previous_link),
            ('next', next_link),
            ('previous_' + self.cursor_query_param, previous_cursor),
            ('next_' + self.cursor_query_param, next_cursor),
        ]))
