import collections

from django.db import transaction

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.callcenter.serializers import RequestUserSerializer
from cars.users.models import User, UserPhoneBinding

from ..core.common_helper import collection_to_mapping
from ..models import RequestUserPhoneBindingEntry


class UserProcessingHelper(object):
    def __init__(self, saas_client):
        self._saas_client = saas_client

    @classmethod
    def from_settings(cls):
        saas_client = SaasDriveAdminClient.from_settings()
        return cls(saas_client)

    def get_entries_phone_bindings(self, entries, *, phone_number_field='phone', use_batch_request=True):
        user_phones_to_disclose = {
            p for p in (getattr(entry, phone_number_field, None) for entry in entries)
            if p is not None
        }

        # make "active" first if exists

        if use_batch_request and len(user_phones_to_disclose) > 1:
            grouped_related_users = collection_to_mapping(
                User.objects.filter(phone__in=user_phones_to_disclose).order_by('phone', 'status'),
                attr_key='phone'
            )
            related_users = {
                p: grouped_related_users.get(p, [None])[0]
                for p in user_phones_to_disclose
            }
        else:
            related_users = {
                p: User.objects.filter(phone=p).order_by('status').first()
                for p in user_phones_to_disclose
            }

        return related_users

    def filter_deleted_users(self, user_mapping, request):
        keys_to_hide = []

        for key, user in user_mapping.items():
            if user is None:
                continue

            if user.status == 'deleted':
                keys_to_hide.append(key)
            elif not self._saas_client.check_access_to_deleting_user(user.id, request):
                keys_to_hide.append(key)
            else:
                pass

        for key in keys_to_hide:
            user_mapping[key] = None

        return user_mapping

    def format_users(self, related_users=None, default=None, update_from_key=None):
        formatted_related_users = collections.defaultdict(lambda: RequestUserSerializer(default).data)

        for key, user in (related_users or {}).items():
            formatted_user = self.format_user(user)

            if update_from_key is not None and formatted_user[update_from_key] is None:
                formatted_user[update_from_key] = str(key)

            formatted_related_users[key] = formatted_user

        return formatted_related_users

    def format_user(self, user):
        formatted_user = RequestUserSerializer(user).data
        return formatted_user


class PhoneBindingHelper(object):
    """
    User phone bindings intended to bind user, phone number and source number has been obtained from

    Main binding entry may be without user, but it's always unique
     and it's either verified or has None source type.

    There could be multiple other entries besides the main one,
     but they only needed to provide extra info and won't be used directly as foreign keys.
    """

    PhoneSourceType = RequestUserPhoneBindingEntry.PhoneSourceType

    def try_bind_phone(self, phone_number):
        assert phone_number, 'phone number is expected to be not empty'

        verified_user_binding, unverified_user_bindings = self._get_binding_entries(phone_number)

        if verified_user_binding is None:
            with transaction.atomic(savepoint=False):
                verified_user_binding = RequestUserPhoneBindingEntry.objects.filter(phone=phone_number).first()

                # to be done: add unique constraint and integrity exception check
                if verified_user_binding is None:
                    verified_user_binding = RequestUserPhoneBindingEntry.objects.create(phone=phone_number)

        verified_user_binding = self._try_update_main_user_binding(
            phone_number, verified_user_binding, unverified_user_bindings
        )

        return verified_user_binding, unverified_user_bindings

    def _get_binding_entries(self, phone_number):
        verified_user_bindings, unverified_user_bindings = [], []

        all_user_bindings = (
            RequestUserPhoneBindingEntry.objects
            .filter(phone=phone_number)
        )

        for b in all_user_bindings:
            c = verified_user_bindings if b.is_verified or b.phone_source is None else unverified_user_bindings
            c.append(b)

        if len(verified_user_bindings) > 1:
            raise Exception('multiple main entries are not supported (phone number is {})'
                            .format(phone_number))
        else:
            verified_user_binding = next(iter(verified_user_bindings), None)

        return verified_user_binding, unverified_user_bindings

    def _get_actual_users(self, bindings):
        return {b.user for b in bindings if b.user is not None}

    def _try_update_main_user_binding(self, phone_number, verified_user_binding, unverified_user_bindings):
        if verified_user_binding.can_be_rebound:
            user, phone_source = self._try_get_user_entry(phone_number, unverified_user_bindings)

            if user is not None:
                verified_user_binding.user = user
                verified_user_binding.phone_source = phone_source
                verified_user_binding.save()

        return verified_user_binding

    def _try_get_user_entry(self, phone_number, unverified_user_bindings):
        user = self._try_get_direct_user(phone_number)
        phone_source = self.PhoneSourceType.DIRECT_USER.value

        if user is None:
            user = self._try_get_bound_user(phone_number)
            phone_source = self.PhoneSourceType.USER_BINDING.value

        if user is None:
            user = self._try_get_extra_tag_user(unverified_user_bindings)
            phone_source = self.PhoneSourceType.CALL_TAG_BINDING.value

        if user is None:
            user = self._try_get_extra_chat_user(unverified_user_bindings)
            phone_source = self.PhoneSourceType.CHAT_BINDING.value

        return user, phone_source

    def _try_get_direct_user(self, phone_number):
        user_with_requested_phone = User.objects.filter(
            phone=phone_number
        ).first()
        return user_with_requested_phone

    def _try_get_bound_user(self, phone_number):
        existing_user_phone_binding = (
            UserPhoneBinding.objects
            .select_related('user')
            .filter(phone=phone_number)
            .order_by('-submit_date')
            .first()
        )

        bound_user = existing_user_phone_binding.user if existing_user_phone_binding is not None else None
        return bound_user

    def _try_get_extra_tag_user(self, unverified_user_bindings):
        return self._try_get_filtered_extra_user(
            unverified_user_bindings,
            self.PhoneSourceType.CALL_TAG_EXTRA_BINDING.value
        )

    def _try_get_extra_chat_user(self, unverified_user_bindings):
        return self._try_get_filtered_extra_user(
            unverified_user_bindings,
            self.PhoneSourceType.CHAT_EXTRA_BINDING.value
        )

    def _try_get_filtered_extra_user(self, unverified_user_bindings, phone_source_filter):
        extra_users = {
            b.user for b in unverified_user_bindings
            if b.phone_source == phone_source_filter
        }

        if len(extra_users) == 1:
            return next(iter(extra_users))

        return None

    def try_bind_chat_extra_phone(self, related_client, phone_number):
        """Try bind phone obtained from ambiguous source (chat text here) to a related client

        Note: related client is a main chat2desk client entry. Additional entries have no user binding.
        """
        verified_user_binding, unverified_user_bindings = self.try_bind_phone(phone_number)

        existing_binding = related_client.user_binding
        existing_user = existing_binding.user if existing_binding is not None else None

        verified_user_binding = self._try_make_extra_entry(
            phone_number, existing_user, self.PhoneSourceType.CHAT_EXTRA_BINDING.value,
            verified_user_binding, unverified_user_bindings
        )

        if existing_binding is None and verified_user_binding is not None:
            related_client.user_binding = verified_user_binding
            related_client.save()

        return verified_user_binding, unverified_user_bindings

    def try_bind_call_tag_extra_phone(self, possible_phone, phone_number):
        verified_user_binding, unverified_user_bindings = self.try_bind_phone(possible_phone)

        if possible_phone != phone_number:
            verified_existing_user_binding, _ = self.try_bind_phone(phone_number)

            existing_user = verified_existing_user_binding.user
            verified_user_binding = self._try_make_extra_entry(
                phone_number, existing_user, self.PhoneSourceType.CALL_TAG_EXTRA_BINDING.value,
                verified_user_binding, unverified_user_bindings
            )

        return verified_user_binding, unverified_user_bindings

    def _try_make_extra_entry(self, phone_number, existing_user, phone_source,
                              verified_user_binding, unverified_user_bindings):
        if (
                existing_user is not None and
                (verified_user_binding.user is None or verified_user_binding.user != existing_user)
        ):
            # create an extra entry just to save binding trial info
            extra_binding = RequestUserPhoneBindingEntry.objects.create(
                phone=phone_number,
                user=existing_user,
                phone_source=phone_source,
            )
            unverified_user_bindings.append(extra_binding)

            verified_user_binding = self._try_update_main_user_binding(
                phone_number, verified_user_binding, unverified_user_bindings
            )

        return verified_user_binding
