import enum
from collections import namedtuple
import uuid

import cars.settings

from cars.admin.models import PushMessagesHistory
from cars.callcenter.core import StaffInfoHelper
from cars.callcenter.serializers.user import StaffUserEntrySerializer
from cars.carsharing.models import TagDescription
from cars.core.util import datetime_helper
from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.users.models import User, UserTagHistory

from cars.request_aggregator.core.phone_binding_helper import UserProcessingHelper
from cars.request_aggregator.serializers.common import AggregatedResponseEntry, RequestType


NotificationTagData = namedtuple('NotificationTagData',
                                 ['queue', 'request_type', 'message_text'])


class PushStatus(enum.Enum):
    SERVICED = 'serviced'
    NOT_SERVICED = 'not_serviced'

    @classmethod
    def make(cls, is_delivered):
        return cls.SERVICED if is_delivered else cls.NOT_SERVICED


class BasePushHistoryHelper(object):
    REQUEST_TYPE = RequestType.push
    PUSH_QUEUE_NAME = 'push'

    user_processing_helper = UserProcessingHelper.from_settings()
    saas_client = SaasDriveAdminClient.from_settings()
    staff_info_helper = StaffInfoHelper.make_default()

    def __init__(self, denied_operator_view_roles):
        self._denied_operator_view_roles = denied_operator_view_roles

    @classmethod
    def from_settings(cls):
        denied_operator_view_roles = cars.settings.REQUEST_AGGREGATOR['callcenter']['denied_operator_view_roles']
        return cls(denied_operator_view_roles)

    def check_request_type(self, request_type_collection=None):
        return not request_type_collection or self.REQUEST_TYPE.value in request_type_collection

    def get_requests(self, request=None, limit=None, **kwargs):
        raise NotImplementedError

    def _check_user_not_matched(self, user=None, phone_number=None, **kwargs):
        # phone number specified may be not bound to a user
        return phone_number is not None and user is None

    def _make_access_mapping(self, request=None):
        access_mapping = {}

        if request is not None:
            has_access_to_operators = not self.saas_client.check_user_role(
                *self._denied_operator_view_roles, request=request, require_all=False
            )
        else:
            has_access_to_operators = False

        access_mapping['has_access_to_operators'] = has_access_to_operators

        return access_mapping


class OrdinaryPushHistoryHelper(BasePushHistoryHelper):
    def get_requests(self, request=None, limit=None, **kwargs):
        return []

        if self._check_user_not_matched(**kwargs):
            return []

        filters = self._make_filters(**kwargs)

        if not filters and limit is None:
            raise Exception('at least one filter must be provided')

        messages = (
            PushMessagesHistory.objects
            .using(cars.settings.DB_RO_ID)
            .select_related('message')
            .prefetch_related('sender')
            .filter(**filters)
            .order_by('-time_id')
            [:limit]
        )

        senders = {
            m.sender: self.staff_info_helper.get_agent_entry(user_id=str(m.sender.id))
            for m in messages
            if m.sender is not None
        }

        users_mapping = self._make_users_mapping(messages, request, kwargs)

        access_mapping = self._make_access_mapping(request)

        response_entries = [
            self._process_entry(m, senders, users_mapping, access_mapping)._asdict()
            for m in messages
        ]
        return response_entries

    def _make_filters(self, since=None, until=None, user=None, staff_entry_binding=None, **kwargs):
        filters = {}

        if user is not None:
            filters['uid'] = user.uid

        if since is not None:
            filters['time_id__gte'] = since

        if until is not None:
            filters['time_id__lt'] = until

        if staff_entry_binding is not None:
            filters['sender'] = staff_entry_binding.user

        return filters

    def _make_users_mapping(self, messages, request, kwargs):
        if 'user' in kwargs:
            user = kwargs['user']
            users_mapping = {user.uid: user}
        else:
            user_uids_to_disclose = {m.uid for m in messages}
            related_users = User.objects.using(cars.settings.DB_RO_ID).filter(uid__in=user_uids_to_disclose)
            users_mapping = {
                related_user.uid: related_user
                for related_user in related_users
            }

        self.user_processing_helper.filter_deleted_users(users_mapping, request)

        formatted_users_mapping = self.user_processing_helper.format_users(users_mapping)

        return formatted_users_mapping

    def _process_entry(self, m, senders, users_mapping, access_mapping):
        time_enter = time_exit = datetime_helper.datetime_to_timestamp(m.time_id)

        sender = m.sender

        agent_to_format = None

        if sender is not None:
            staff_sender = senders[sender]
            agent_to_format = staff_sender if staff_sender is not None else sender

        agent = StaffUserEntrySerializer(agent_to_format).data

        if access_mapping['has_access_to_operators']:
            operators = [agent]
        else:
            operators = []

        related_user_uid = m.uid
        user = users_mapping.get(related_user_uid, None)

        request_description = {
            'id': None,
            'origin': None,
            'queue': self.PUSH_QUEUE_NAME,
            'type': self.REQUEST_TYPE.value,
        }

        is_delivered = m.is_successfully_delivered
        status = PushStatus.make(is_delivered)

        message = m.message.message
        data = {'message': message}

        entry = AggregatedResponseEntry(
            user=user,
            time_enter=time_enter,
            time_exit=time_exit,
            time_connect=None,
            connect_trial_count=None,
            duration=None,
            duration_print='',
            status=status.value,
            operators=operators,
            request=request_description,
            phone=None,
            data=data,
            data_url=None,
            message=message,
            tags=[],
        )
        return entry


def get_tag_data(meta, type_id):
    if type_id == 'user_push':
        return NotificationTagData(
                'push', RequestType.push,
                meta.get('push_text') or meta.get('message_text', ''))
    elif type_id == 'user_sms':
        return NotificationTagData(
            'sms', RequestType.sms, meta.get('message_text', ''))
    elif type_id == 'user_mail_notification_tag':
        return NotificationTagData(
                'mail', RequestType.mail, meta.get('template_id', ''))
    raise RuntimeError('Unexpected tag type: {}'.format(type_id))


class TagNotificationsHistoryHelper(BasePushHistoryHelper):
    _tag_notifications = None
    NOTIFICATIONS_TAGS_TYPES = (
        'user_sms',
        'user_push',
        'user_mail_notification_tag'
    )

    @property
    def tag_notifications(self):
        if self._tag_notifications is None:
            tags = (
                TagDescription.objects.using(cars.settings.DB_RO_ID)
                .filter(meta__isnull=False)
                .values_list('name', 'meta', 'type')
            )

            self.__class__._tag_notifications = {
                name: (meta, type_id) for name, meta, type_id in tags
                if type_id in self.NOTIFICATIONS_TAGS_TYPES
            }

        return self._tag_notifications

    def get_requests(self, request=None, limit=None, **kwargs):
        return []

        if self._check_user_not_matched(**kwargs):
            return []

        filters = self._make_filters(**kwargs)

        if not filters and limit is None:
            raise Exception('at least one filter must be provided')

        entries = (
            UserTagHistory.objects
            .using(cars.settings.DB_RO_ID)
            .filter(**filters)
            .filter(tag__in=self.tag_notifications.keys(), history_action='remove')
            .order_by('-history_timestamp')
            .only('history_user_id', 'history_timestamp', 'object_id', 'tag')
            [:limit]
        )

        users_mapping = self._make_users_mapping(entries, request)

        access_mapping = self._make_access_mapping(request)

        response_entries = [
            self._process_entry(e, users_mapping, access_mapping)._asdict()
            for e in entries
        ]
        return response_entries

    def _make_filters(self, since=None, until=None, user=None, staff_entry_binding=None, **kwargs):
        filters = {}

        if user is not None:
            filters['object'] = user

        if since is not None:
            filters['history_timestamp__gte'] = datetime_helper.datetime_to_timestamp(since)

        if until is not None:
            filters['history_timestamp__lt'] = datetime_helper.datetime_to_timestamp(until)

        if staff_entry_binding is not None:
            filters['history_user_id'] = str(staff_entry_binding.user.id)

        return filters

    def _make_users_mapping(self, entries, request):
        related_users_id_collection = set()

        for e in entries:
            sender_id = self._cast_user_id(e.history_user_id)
            if sender_id is not None:
                related_users_id_collection.add(sender_id)

            user_id = self._cast_user_id(e.object_id)
            if user_id is not None:
                related_users_id_collection.add(user_id)

        related_users = User.objects.using(cars.settings.DB_RO_ID).filter(id__in=related_users_id_collection)
        related_users_mapping = {str(u.id): u for u in related_users}

        self.user_processing_helper.filter_deleted_users(related_users_mapping, request)

        return related_users_mapping

    def _cast_user_id(self, user_id):
        try:
            if not isinstance(user_id, uuid.UUID):
                user_id = uuid.UUID(user_id)
            value = str(user_id)
        except (TypeError, ValueError):
            value = None
        return value

    def _process_entry(self, entry, related_users_mapping, access_mapping):
        sender_id = entry.history_user_id

        agent_to_format = None

        if sender_id in related_users_mapping:
            sender_entry = related_users_mapping.get(sender_id, None)

            if sender_entry is not None:
                staff_sender_entry = self.staff_info_helper.get_agent_entry(user_id=str(sender_entry.id))
            else:
                staff_sender_entry = None

            agent_to_format = staff_sender_entry if staff_sender_entry is not None else sender_entry

        agent = StaffUserEntrySerializer(agent_to_format).data

        if access_mapping['has_access_to_operators']:
            operators = [agent]
        else:
            operators = []

        related_user_id = str(entry.object_id)
        related_user = related_users_mapping.get(related_user_id, None)
        formatted_user = self.user_processing_helper.format_user(related_user)

        time_enter = time_exit = entry.history_timestamp

        tag_data = get_tag_data(*self.tag_notifications[entry.tag])
        data = {'message': tag_data.message_text}

        request_description = {
            'id': None,
            'origin': None,
            'queue': tag_data.queue,
            'type': tag_data.request_type.value,
        }

        entry = AggregatedResponseEntry(
            user=formatted_user,
            time_enter=time_enter,
            time_exit=time_exit,
            time_connect=None,
            connect_trial_count=None,
            duration=None,
            duration_print='',
            status='serviced',
            operators=operators,
            request=request_description,
            message=tag_data.message_text,
            data_url=None,
            data=data,
            phone=None,
            tags=[],
        )
        return entry
