import collections
import logging
import uuid

from django.db import transaction

from cars.callcenter.core import StaffInfoHelper
from cars.core.util import datetime_helper, phone_number_helper

from cars.settings import REQUEST_AGGREGATOR as settings

from cars.request_aggregator.core import YTHelper
from cars.request_aggregator.core.request_time_sync_helper import RequestEntryCountSyncHelper
from cars.request_aggregator.models.call_tags import RequestTagEntry, TagOrigin, RequestTagType
from cars.request_aggregator.models.call_center_common import SyncOrigin

from .history_helper import RequestTagsHistoryManager
from .description_helper import TagDescriptionHelper


LOGGER = logging.getLogger(__name__)


class RequestTagCollectingSyncHelper(RequestEntryCountSyncHelper):
    @classmethod
    def from_settings(cls, sync_status_origin):
        return cls(
            stat_sync_origin=sync_status_origin,
            max_entries_batch=5000,
        )


class RequestTagCollectingHelper(object):
    TagEntry = collections.namedtuple(
        'TagEntry', (
            'callback', 'comment', 'request', 'result', 'submitTs', 'workerId',
            'model', 'OS', 'airport'  # may be absent (set None)
        )
    )

    TAG_TYPE = RequestTagType.OLD.value

    MAX_ENTRIES_TO_UPDATE_PERFORMER = 5000

    staff_info_helper = StaffInfoHelper.make_default()

    def __init__(self, tag_origin, table_path=None):
        self._tag_origin = tag_origin

        sync_status_origin = self._get_sync_status_origin(tag_origin)
        self._sync_helper = RequestTagCollectingSyncHelper.from_settings(sync_status_origin)

        if table_path is None:
            table_path = self._get_table_path_from_status_origin(tag_origin)

        self._yt_helper = YTHelper(table_path)

        self._call_tag_category_helper = TagDescriptionHelper()

        self._history_manager = RequestTagsHistoryManager()

    @property
    def tag_origin(self):
        return self._tag_origin

    def _get_sync_status_origin(self, tag_origin):
        _mapping = {
            TagOrigin.CARSHARING: SyncOrigin.CC_INTERNAL_CARSHARING_UPDATE_TAGS,
            TagOrigin.CARSHARING_VIP: SyncOrigin.CC_INTERNAL_CARSHARING_VIP_UPDATE_TAGS,
            # TagOrigin.CARSHARING_OUTGOING: SyncOrigin.CC_INTERNAL_OUTGOING_UPDATE_TAGS,  # deprecated
            TagOrigin.CC_INTERNAL_ALTAY_OUTGOING: SyncOrigin.CC_INTERNAL_ALTAY_OUTGOING_UPDATE_TAGS,
            TagOrigin.AUDIOTELE_INCOMING: SyncOrigin.CC_AUDIOTELE_INCOMING_TAGS_UPDATE,
            TagOrigin.AUDIOTELE_OUTGOING: SyncOrigin.CC_AUDIOTELE_OUTGOING_TAGS_UPDATE,
        }

        if not isinstance(tag_origin, TagOrigin) or tag_origin not in _mapping:
            raise Exception('invalid tag origin: {}'.format(tag_origin))

        return _mapping[tag_origin]

    def _get_table_path_from_status_origin(self, tag_origin):
        _mapping = {
            TagOrigin.CARSHARING: settings['tags']['internal_cc_carsharing_table'],
            TagOrigin.CARSHARING_VIP: settings['tags']['internal_cc_carsharing_vip_table'],
            # TagOrigin.CARSHARING_OUTGOING: settings['tags']['internal_cc_outgoing_table'],  # deprecated
            TagOrigin.CC_INTERNAL_ALTAY_OUTGOING: settings['tags']['internal_cc_outgoing_table'],
            TagOrigin.AUDIOTELE_INCOMING: settings['tags']['audiotele_incoming_table'],
            TagOrigin.AUDIOTELE_OUTGOING: settings['tags']['audiotele_outgoing_table'],
        }

        if not isinstance(tag_origin, TagOrigin) or tag_origin not in _mapping:
            raise Exception('invalid tag origin: {}'.format(tag_origin))

        return _mapping[tag_origin]

    def update_data(self):
        total_count = self._yt_helper.get_table_row_count()

        updated_count = 0

        for start_index, end_index in self._sync_helper.iter_indexes_to_process(total_count):
            entries_to_create = [
                self._make_request_tag_entry(entry)
                for entry in self._yt_helper.import_data(start_index, end_index)
            ]

            with transaction.atomic(savepoint=False):
                RequestTagEntry.objects.bulk_create(entries_to_create)
                self._history_manager.bulk_add_entries(entries_to_create)

            updated_count += (end_index - start_index)

        return updated_count

    def _make_request_tag_entry(self, entry):
        entry = self._make_tag_entry(entry)

        original_phone = entry.callback
        phone_number = phone_number_helper.normalize_phone_number(original_phone)

        submit_timestamp_ms = entry.submitTs  # in milliseconds
        submitted_at = datetime_helper.timestamp_ms_to_datetime(submit_timestamp_ms)

        request = self._decode_string(entry.request)  # printable general category
        result = self._decode_string(entry.result)  # detailed category
        tag_entry = self._call_tag_category_helper.get_or_create_old_tag_category_by_traits(
            request=request, result=result, tag_origin=self._tag_origin.value
        )

        worker_id = str(uuid.UUID(entry.workerId)) if entry.workerId else None
        performer_staff_entry = self.staff_info_helper.get_agent_entry(yang_worker_id=worker_id)
        performer_user_entry = performer_staff_entry.user if performer_staff_entry is not None else None

        comment = self._decode_string(entry.comment)

        meta_info = self._filter_mapping(
            # do not include original fields if converted ones are OK
            original_phone=(original_phone if phone_number is None else None),
            worker_id=(worker_id if performer_staff_entry is None else None),
            model=entry.model,
            airport=entry.airport,
            OS=entry.OS,
        )

        return RequestTagEntry(
            submitted_at=submitted_at,
            performer=performer_user_entry,
            tag_id=tag_entry.id,
            tag_type=self.TAG_TYPE,
            request_id=None,  # a daemon performs binding
            request_origin=self._tag_origin.value,
            original_phone=phone_number,
            original_user_id=None,  # a daemon performs binding
            comment=comment,
            meta_info=meta_info,
        )

    def _make_tag_entry(self, entry):
        flattened_entry = (entry.get(name, None) for name in self.TagEntry._fields)
        tag_entry = self.TagEntry._make(flattened_entry)

        if len(entry) != len(tag_entry):
            absent_keys = set(entry.keys()).difference(self.TagEntry._fields)
            if absent_keys:
                LOGGER.warning('entry has unexpected fields: {}'.format(list(absent_keys)))

        return tag_entry

    def _decode_string(self, value):
        try:
            if value is None:
                return ''
            return bytes(value, 'iso-8859-1').decode('utf-8')
        except UnicodeEncodeError:
            return value

    def _filter_mapping(self, **kwargs):
        return {k: v for k, v in kwargs.items() if v}

    def update_performer_info(self, offset=0, count=None):
        count = min(count or self.MAX_ENTRIES_TO_UPDATE_PERFORMER, self.MAX_ENTRIES_TO_UPDATE_PERFORMER)

        processed = -1
        updated = 0

        with transaction.atomic(savepoint=False):
            entries = (
                RequestTagEntry.objects.select_for_update()
                .filter(performer__isnull=True)
                [offset:offset + count]
            )

            for processed, entry in enumerate(entries):
                worker_id = entry.meta_info.get('worker_id', None)
                performer_staff_entry = self.staff_info_helper.get_agent_entry(yang_worker_id=worker_id)
                performer_user_entry = performer_staff_entry.user if performer_staff_entry is not None else None

                if performer_user_entry is not None:
                    entry.performer = performer_user_entry
                    entry.meta_info.pop('worker_id')

                    entry.save()

                    self._history_manager.update_entry(entry)

                    updated += 1

        return updated, (processed + 1)
