import collections
import datetime
import logging
import uuid

from yql.api.v1.client import YqlClient
from yql.client.operation import CantGetResultsException

from django.db import transaction

from cars.callcenter.core import StaffInfoHelper
from cars.core.util import datetime_helper, phone_number_helper

from cars.settings import REQUEST_AGGREGATOR as settings

from cars.request_aggregator.core.request_time_sync_helper import RequestTimeSyncHelper
from cars.request_aggregator.models.call_tags import RequestTagEntry, TagOrigin, RequestTagType
from cars.request_aggregator.models.call_center_common import SyncOrigin

from .history_helper import RequestTagsHistoryManager
from .description_helper import TagDescriptionHelper


LOGGER = logging.getLogger(__name__)


class YTException(Exception):
    pass


class RequestTagYQLCollectingSyncHelper(RequestTimeSyncHelper):
    def __init__(self, stat_sync_origin, *, default_since, max_time_span, table_update_lag):
        super().__init__(stat_sync_origin, default_since=default_since, max_time_span=max_time_span)

        assert isinstance(table_update_lag, datetime.timedelta)
        self._table_update_lag = table_update_lag

    @classmethod
    def from_settings(cls):
        table_update_lag = settings['tags']['yql_import']['table_update_lag']
        return cls(
            stat_sync_origin=SyncOrigin.REQUEST_TAGS_IMPORTING,
            default_since=datetime_helper.utc_localize(datetime.datetime(2019, 3, 31)) + table_update_lag,
            max_time_span=datetime.timedelta(days=3),
            table_update_lag=table_update_lag,
        )

    def iter_time_span_to_process(self):
        for since, until in super().iter_time_span_to_process():
            yield int(since), int(until)

    def _iter_general_time_spans(self, sync_entry):
        table_update_lag_s = self._table_update_lag.total_seconds()

        for since, until in super()._iter_general_time_spans(sync_entry):
            yield since - table_update_lag_s, until - table_update_lag_s


class RequestTagYQLImportingHelper(object):
    YQL_REQUEST_TEMPLATE = """
PRAGMA yt.InferSchema;
PRAGMA yson.DisableStrict;

$get_item = ($arr, $field) -> {{
        $values = Yson::ConvertToStringDict(Yson::ParseJson($arr)[0].output_values);
        return DictLookUp($values, $field);
        }};

select $get_item(assignment_raw_solutions, "OS") as os,
       $get_item(assignment_raw_solutions, "model") as model,
       $get_item(assignment_raw_solutions, "result") as result,
       $get_item(assignment_raw_solutions, "comment") as comment,
       $get_item(assignment_raw_solutions, "request") as request,
       $get_item(assignment_raw_solutions, "callback") as callback,
       $get_item(assignment_raw_solutions, "airport") as airport,
       worker_id, assignment_approve_time, project_id
 from `{table_path}`
 where project_id in ({project_ids}) 
 and assignment_status == "APPROVED"
 and CAST(assignment_approve_time as UINT32) >= {since}
 and CAST(assignment_approve_time as UINT32) < {until}
;
    """

    def __init__(self, token, cluster_name, table_path, project_ids):
        self._token = token
        self._cluster_name = cluster_name
        self._table_path = table_path
        self._project_ids = ', '.join('"{}"'.format(pid) for pid in project_ids)

    @classmethod
    def from_settings(cls):
        yql_import_settings = settings['tags']['yql_import']
        return cls(
            token=yql_import_settings['token'],
            cluster_name=yql_import_settings['cluster'],
            table_path=yql_import_settings['table'],
            project_ids=RequestTagFactory.tag_origin_mapping,
        )

    def import_data(self, since, until):
        prepared_request = self.YQL_REQUEST_TEMPLATE.format(
            table_path=self._table_path,
            project_ids=self._project_ids,
            since=since,
            until=until,
        )

        with YqlClient(db=self._cluster_name, token=self._token) as client:
            request = client.query(prepared_request, syntax_version=1)
            request.run()

            LOGGER.info("YQL shared link: {}".format(request.share_url))

            results = request.get_results()  # blocks until they are ready

            try:
                request_status = request.status
            except CantGetResultsException:
                raise YTException('YQL query seems to be interrupted')

            if request_status == "ABORTED":
                raise YTException('YQL query was interrupted')

            if request_status.endswith('ERROR'):
                LOGGER.error('Error while executing YQL query')
                for error in request.errors:
                    for message in str(error).splitlines():
                        LOGGER.error(message.strip())
                raise YTException('Error in YQL api request: {}'.format(request.errors[0].message))

            table = results.table
            table.fetch_full_data()

            column_names = table.column_names

            for row in table.rows:
                yield dict(zip(column_names, row))


class RequestTagFactory(object):
    TAG_TYPE = RequestTagType.OLD.value

    tag_origin_mapping = {
        '3013': TagOrigin.CARSHARING,
        '3010': TagOrigin.CARSHARING_VIP,
        '3012': TagOrigin.CC_INTERNAL_ALTAY_OUTGOING,
        '3307': TagOrigin.AUDIOTELE_INCOMING,
        '3308': TagOrigin.AUDIOTELE_OUTGOING,
    }

    def __init__(self):
        self._staff_info_helper = StaffInfoHelper.make_default()
        self._call_tag_category_helper = TagDescriptionHelper()

    def make_entry(self, entry):
        meta_info = {}

        original_phone = entry['callback']
        phone_number = phone_number_helper.normalize_phone_number(original_phone)

        if phone_number is None:
            meta_info['original_phone'] = original_phone

        approval_time = entry['assignment_approve_time']
        submitted_at = datetime_helper.timestamp_to_datetime(approval_time)

        tag_origin = self._get_tag_origin(entry['project_id'])

        request = self._decode_string(entry['request'])  # printable general category
        result = self._decode_string(entry['result'])  # detailed category

        tag_entry = self._call_tag_category_helper.get_or_create_old_tag_category_by_traits(
            request=request, result=result, tag_origin=tag_origin.value
        )

        worker_id = str(uuid.UUID(entry['worker_id'])) if entry['worker_id'] else None
        performer_staff_entry = self._staff_info_helper.get_agent_entry(yang_worker_id=worker_id)

        if performer_staff_entry is not None:
            performer_user_id = performer_staff_entry.user_id
        else:
            performer_user_id = None
            meta_info['worker_id'] = worker_id

        comment = self._decode_string(entry['comment'])

        for extra_property in ('model', 'airport', 'os'):
            extra_property_value = entry.get(extra_property, None)
            if extra_property_value:
                meta_info[extra_property] = extra_property_value

        return RequestTagEntry(
            submitted_at=submitted_at,
            performer_id=performer_user_id,
            tag_id=tag_entry.id,
            tag_type=self.TAG_TYPE,
            request_id=None,  # a daemon performs binding
            request_origin=tag_origin.value,
            original_phone=phone_number,
            original_user_id=None,  # a daemon performs binding
            comment=comment,
            meta_info=meta_info,
        )

    def _get_tag_origin(self, project_id):
        return self.tag_origin_mapping[project_id]

    def _decode_string(self, value):
        try:
            if value is None:
                return ''
            return bytes(value, 'iso-8859-1').decode('utf-8')
        except (UnicodeEncodeError, UnicodeDecodeError):
            return value


class RequestTagYQLCollectingHelper(object):
    MAX_ENTRIES_BATCH = 1000

    def __init__(self):
        self._sync_helper = RequestTagYQLCollectingSyncHelper.from_settings()

        self._importing_helper = RequestTagYQLImportingHelper.from_settings()

        self._request_tag_factory = RequestTagFactory()
        self._history_manager = RequestTagsHistoryManager()

    @classmethod
    def from_settings(cls):
        return cls()

    def update_data(self):
        entries_stat = collections.Counter(processed_entries_count=0)

        for since, until in self._sync_helper.iter_time_span_to_process():
            entry_batches, history_entry_batches = [], []

            data = self._importing_helper.import_data(since, until)

            while True:
                entries_to_create = [
                    self._request_tag_factory.make_entry(entry)
                    for _, entry in zip(range(self.MAX_ENTRIES_BATCH), data)
                ]

                if entries_to_create:
                    entry_batches.append(entries_to_create)

                    history_entries = self._history_manager.prepare_history_records(entries_to_create)
                    history_entry_batches.append(history_entries)
                else:
                    break

            with transaction.atomic(savepoint=False):
                for batch, history_batch in zip(entry_batches, history_entry_batches):
                    RequestTagEntry.objects.bulk_create(batch)
                    self._history_manager.bulk_add_prepared_entries(history_batch)
                    entries_stat['processed_entries_count'] += len(batch)

        return entries_stat
