import functools
import logging
import threading
import time

from django.db import transaction

import cars.settings

from cars.knowledge_base.core.category_tree.selection_helper import CategoryTreeSelectionHelper

from cars.request_aggregator.core.common_helper import collection_to_mapping
from cars.request_aggregator.models.call_tags import CallTagCategory, RequestTagEntry, RequestTagType, TagOrigin
from cars.request_aggregator.serializers.request_tags import RequestTagSerializer

LOGGER = logging.getLogger(__name__)


class TagDescriptionHelper(object):
    COMMENT_MAX_PRINT_LENGTH = RequestTagEntry.COMMENT_MAX_PRINT_LENGTH
    INVALIDATE_PERIOD_S = 60

    in_request_translation_mapping = {
        'attachment': 'Приложение',
        'checkin': 'Регистрация',
        'complaints': 'Жалобы',
        'critical': 'Критичное',
        'general': 'Общая информация о сервисе',
        'offers': 'Предложения',
        'other': 'Другое',
        'trip': 'Поездка',
        'tripPC': 'Бортовой компьютер',
    }

    out_request_translation_mapping = {
        'evacuation': 'Эвакуация',
        'other': 'Другое',
        'rent': 'Длительная аренда',
        'speed': 'Превышение скорости',
        'thigs': 'Забытые вещи',
        'violations': 'Нарушения',
    }

    def __init__(self):
        self._cached_tree = None

        if cars.settings.CALLCENTER['tag_description_invalidation_enabled']:
            self._invalidation_worker = threading.Thread(target=self._invalidate_runner, daemon=True)
            self._invalidation_worker.start()

    @property
    def cached_tree(self):
        if self._cached_tree is None:
            self._invalidate_cache_tree()
        return self._cached_tree

    def _invalidate_cache_tree(self):
        self._cached_tree = CategoryTreeSelectionHelper().build_tree()

    def invalidate(self):
        self._invalidate_cache_tree()
        self.get_tag_category_entry.cache_clear()
        self.get_all_tag_categories.cache_clear()
        LOGGER.debug('tag description helper cache has been invalidated')

    def _invalidate_runner(self):
        while True:
            time.sleep(self.INVALIDATE_PERIOD_S)
            self.invalidate()

    def get_call_tags(self, request_id_collection, request_origin_filter):
        entry_tags = RequestTagEntry.objects.filter(
            request_origin_filter,
            request_id__in=request_id_collection
        )
        entry_tags_mapping = collection_to_mapping(entry_tags, attr_key='request_id', value=self._get_description)
        return entry_tags_mapping

    def _get_description(self, tag_entry):
        assert isinstance(tag_entry, RequestTagEntry)
        formatted_entry = RequestTagSerializer(instance=tag_entry, context={'tag_description_helper': self}).data
        return formatted_entry

    @functools.lru_cache(maxsize=None)
    def get_tag_category_entry(self, tag_id, tag_type):
        if tag_type == RequestTagType.OLD.value:
            return CallTagCategory.objects.get(id=tag_id)
        elif tag_type == RequestTagType.NEW.value:
            return self.cached_tree.get_node(tag_id)
        raise NotImplementedError

    def get_tag_category_description(self, entry, tag_type, locale='ru'):
        if tag_type == RequestTagType.OLD.value:
            description = [(locale == 'ru' and entry.request_ru) or entry.request, entry.result]
        elif tag_type == RequestTagType.NEW.value:
            path = self.cached_tree.find_node_path(entry['id'])

            raw_description = [
                x['meta_info']['labels'].get(locale) or x['meta_info']['labels'].get('ru')
                for x in path
            ]

            if len(raw_description) == 1:
                description = [raw_description[0], raw_description[0]]
            elif len(raw_description) > 2:
                description = [raw_description[0], ' | '.join(raw_description[1:])]
            else:
                description = raw_description
        else:
            raise NotImplementedError

        return description

    def get_tag_category_order(self, entry, tag_type):
        if tag_type == RequestTagType.OLD.value:
            order = [0]
        elif tag_type == RequestTagType.NEW.value:
            path = self.cached_tree.find_node_path(entry['id'])
            order = [x['meta_info'].get('order', 0) for x in path]
        else:
            raise NotImplementedError
        return order

    @functools.lru_cache(maxsize=None)
    def get_or_create_old_tag_category_by_traits(self, request, result, tag_origin):
        with transaction.atomic(savepoint=False):
            tag_entry = CallTagCategory.objects.filter(
                request=request, result=result, tag_origin=tag_origin
            ).first()

            # to be done: add unique constraint and integrity exception check
            if tag_entry is None:
                tag_entry = CallTagCategory.objects.create(
                    request=request, result=result, tag_origin=tag_origin
                )

        if tag_entry.request_ru is None:
            tag_origin_instance = TagOrigin(tag_origin)
            if tag_origin_instance.is_incoming_call:
                request_ru = self.in_request_translation_mapping.get(request, None)
            else:
                request_ru = self.out_request_translation_mapping.get(request, None)

            if request_ru is not None:
                tag_entry.request_ru = request_ru
                tag_entry.save()

        return tag_entry

    @functools.lru_cache(maxsize=None)
    def get_all_tag_categories(self, tag_origin=None, tag_type=None, enabled_only=True):
        if tag_type == RequestTagType.OLD.value:
            entries = CallTagCategory.objects.all()
            if tag_origin is not None:
                entries = entries.filter(tag_origin=tag_origin)
        elif tag_type == RequestTagType.NEW.value:
            tag_origin_instance = TagOrigin(tag_origin)

            if tag_origin_instance.is_incoming_call:
                direction = 'incoming'
            elif tag_origin_instance.is_outgoing_call:
                direction = 'outgoing'
            else:
                raise NotImplementedError

            entries = self.cached_tree.filter_nodes(
                lambda n: (
                    n['meta_info'].get('call_direction') == direction and
                    not n['children']
                )
            )

            if enabled_only:
                entries = (
                    e for e in entries if all(
                        n['meta_info'].get('enabled', True)
                        for n in self.cached_tree.find_node_path(e['id'])
                    )
                )
        else:
            raise NotImplementedError

        entries = list(entries)
        return entries
