import datetime
import json
import logging

from django.db.models import Max

import cars.settings

from cars.core.util import datetime_helper
from cars.core.history import HistoryAction

from cars.callcenter.core.staff_helper import StaffInfoHelper

from cars.request_aggregator.core.common_helper import YTHelper
from cars.request_aggregator.core.request_time_sync_helper import RequestStateSyncHelper
from cars.request_aggregator.core.request_tags import TagDescriptionHelper
from cars.request_aggregator.models.call_tags import (
    RequestTagEntry, RequestTagEntryHistory, RequestOriginType
)


LOGGER = logging.getLogger(__name__)


class RequestTagExportingSyncHelper(RequestStateSyncHelper):
    MAX_ENTRIES_COUNT = 5000

    @classmethod
    def from_settings(cls, sync_status_origin):
        return cls(
            stat_sync_origin=sync_status_origin,
            default_state={'next_history_entry_id': 1}
        )

    def _get_state(self, sync_entry):
        history_entry_id_lo = sync_entry.data['next_history_entry_id']

        is_default = (history_entry_id_lo == 1)

        aggregate_qs = RequestTagEntryHistory.objects.using(cars.settings.DB_RO_ID).aggregate(Max('history_event_id'))
        max_history_entry_id = aggregate_qs.get('history_event_id__max', 0)

        history_entry_id_hi = min(history_entry_id_lo + self.MAX_ENTRIES_COUNT, max_history_entry_id + 1)

        return history_entry_id_lo, history_entry_id_hi, is_default

    def _update_state(self, sync_entry, state=None):
        history_entry_id_lo, history_entry_id_hi, is_default = state

        sync_entry.data['next_history_entry_id'] = history_entry_id_hi

        # try to get more entries
        has_more = (history_entry_id_hi - history_entry_id_lo) == self.MAX_ENTRIES_COUNT
        return has_more


class RequestTagExportingHelper(object):
    # tag history has been forcefully cleared before the date specified
    MIN_SUBMIT_DATE = datetime_helper.utc_localize(datetime.datetime(2018, 4, 13))
    MIN_HISTORY_DATE = datetime_helper.utc_localize(datetime.datetime(2019, 3, 31))

    MAX_ENTRIES_COUNT = RequestTagExportingSyncHelper.MAX_ENTRIES_COUNT

    def __init__(self, sync_origin, tag_origin, export_table_path, add_request_origin=False):
        self._staff_helper = StaffInfoHelper.make_default()

        self._sync_helper = RequestTagExportingSyncHelper.from_settings(sync_origin)
        self._yt_helper = YTHelper(export_table_path)

        self._tag_origin = tag_origin
        self._add_request_origin = add_request_origin

        self._tag_description_helper = TagDescriptionHelper()

    def perform(self):
        for sync_state in self._sync_helper.iter_states_to_process():
            data_to_store = self._iter_data_to_store(sync_state)
            self._yt_helper.export_data(data_to_store)

    def optimize(self):
        self._yt_helper.merge_chunks()

    def _iter_data_to_store(self, sync_state):
        history_event_id_lo, history_event_id_hi, is_default = sync_state

        if is_default:
            entries = self._iter_default_entries()
            yield from (self._make_export_entry(entry) for entry in entries)

        if history_event_id_lo < history_event_id_hi:
            entries = self._iter_history_entries(history_event_id_lo, history_event_id_hi)
            yield from (self._make_export_entry(entry) for entry in entries)

    def _iter_default_entries(self):
        date = self.MIN_SUBMIT_DATE

        while date < self.MIN_HISTORY_DATE:
            next_date = date + datetime.timedelta(days=1)

            entries = RequestTagEntry.objects.using(cars.settings.DB_RO_ID).filter(
                submitted_at__gte=date,
                submitted_at__lt=next_date,
            )

            if self._tag_origin is not None:
                entries = entries.filter(request_origin=self._tag_origin.value)

            yield from entries

            date = next_date

    def _iter_history_entries(self, history_event_id_lo, history_event_id_hi):
        entries = RequestTagEntryHistory.objects.using(cars.settings.DB_RO_ID).filter(
            history_event_id__gte=history_event_id_lo,
            history_event_id__lt=history_event_id_hi,
            history_action=HistoryAction.ADD.value,
        )

        if self._tag_origin is not None:
            entries = entries.filter(request_origin=self._tag_origin.value)

        yield from entries

    def _make_export_entry(self, entry):
        def try_to_str(value):
            return str(value) if value is not None else None

        if entry.original_phone:
            original_phone = str(entry.original_phone)
        elif entry.meta_info:
            original_phone = entry.meta_info.get('original_phone', None)
        else:
            original_phone = None

        try:
            tag_entry = self._tag_description_helper.get_tag_category_entry(entry.tag_id, entry.tag_type)
        except Exception:
            LOGGER.exception('error getting tag entry: id - {}, type - {}'.format(entry.tag_id, entry.tag_type))
            request = request_ru = result = result_ru = ''
        else:
            request, result = self._tag_description_helper.get_tag_category_description(
                tag_entry, entry.tag_type, locale='en'
            )
            request_ru, result_ru = self._tag_description_helper.get_tag_category_description(
                tag_entry, entry.tag_type, locale='ru'
            )

        submit_timestamp = int(entry.submitted_at.timestamp() * 1000)

        worker_id = None
        performer_username = None

        if entry.performer_id is not None:
            agent_entry = self._staff_helper.get_agent_entry(user_id=entry.performer_id)
            if agent_entry is not None:
                worker_id = agent_entry.yang_worker_id
                performer_username = agent_entry.username

        if not worker_id and entry.meta_info:
            worker_id = entry.meta_info.get('worker_id', None)

        if worker_id:
            worker_id = str(worker_id).replace('-', '')  # use custom uuid format without dashes

        data = {
            'callback': original_phone,
            'comment': entry.comment,
            'request': request,
            'result': result,
            'submitTs': submit_timestamp,
            'workerId': worker_id,

            'request_ru': request_ru,
            'result_ru': result_ru,

            'entryId': str(entry.entry_id),
            'tagId': str(entry.tag_id),
            'tagType': str(entry.tag_type),
            'requestId': try_to_str(entry.request_id),
            'requestOrigin': RequestOriginType(entry.request_origin).name,
            'originalPhone': original_phone,
            'originalUserId': try_to_str(entry.original_user_id),
            'relatedPhone': try_to_str(entry.related_phone),
            'relatedUserId': try_to_str(entry.related_user_id),
            'performerId': try_to_str(entry.performer_id),
            'performerUsername': performer_username,
            'metaInfo': json.dumps(entry.meta_info) if entry.meta_info is not None else None,
        }

        return data
