import datetime
import logging
import uuid

from django.db import transaction

import cars.settings

from cars.core.util import datetime_helper
from cars.callcenter.core import StaffInfoHelper

from cars.request_aggregator.core.common_helper import collection_to_mapping

from cars.request_aggregator.models.call_center_common import SyncOrigin
from cars.request_aggregator.models.call_tags import RequestTagEntry, RequestOriginType, RequestTagType

from ..request_time_sync_helper import RequestTimeSyncHelper
from .history_helper import RequestTagsHistoryManager
from .description_helper import TagDescriptionHelper


LOGGER = logging.getLogger(__name__)


class TagBindingTimeSyncHelper(RequestTimeSyncHelper):
    @classmethod
    def from_settings(cls, stat_sync_origin):
        # collect 1 hour time range each time, but not more than 4 hours if error occurred
        # take care about tags uploading lag
        return cls(
            stat_sync_origin,
            default_since=datetime_helper.utc_localize(datetime.datetime(2018, 2, 14)),
            max_time_span=datetime.timedelta(hours=4),
            request_lag=datetime.timedelta(hours=3),
            use_timestamp=False,
        )


class RequestTagBindingHelper(object):
    staff_info_helper = StaffInfoHelper.make_default()

    def __init__(self, stat_sync_origin, request_origin):
        assert isinstance(stat_sync_origin, SyncOrigin)
        assert isinstance(request_origin, RequestOriginType)

        self._time_sync_helper = TagBindingTimeSyncHelper.from_settings(stat_sync_origin)
        self._request_origin = request_origin.value

    def update_tag_bindings(self):
        LOGGER.info('update tag bindings: {}'.format(self.__class__.__name__))

        for since, until in self._time_sync_helper.iter_time_span_to_process():
            self._update_tag_bindings(since, until)

    @transaction.atomic(savepoint=False)
    def _update_tag_bindings(self, since, until):
        entries_to_process = self._get_entry_to_process_values(since, until)

        LOGGER.info(
            'processing {} entries to bind tags since {} until {}'
            .format(len(entries_to_process), since, until)
        )

        if not entries_to_process:
            return

        already_bound_entries = (
            RequestTagEntry.objects
            .using(cars.settings.DB_RO_ID)
            .filter(
                request_origin=self._request_origin,
                request_id__in=(t['id'] for t in entries_to_process),
            )
            .values('entry_id', 'request_id')
        )

        bound_entry_ids = {e['request_id'] for e in already_bound_entries}
        bound_tag_ids = {e['entry_id'] for e in already_bound_entries}

        entries_to_bind = [e for e in entries_to_process if e['id'] not in bound_entry_ids]

        LOGGER.info('total entries not bound yet: {}'.format(len(entries_to_bind)))

        if not entries_to_bind:
            return

        tag_entries = (
            RequestTagEntry.objects
            .select_for_update()
            .filter(
                request_id__isnull=True,
                request_origin=self._request_origin,
                submitted_at__range=(since, until + self._time_sync_helper.request_lag),
                original_phone__isnull=False,
            )
            .exclude(
                entry_id__in=bound_tag_ids,
            )
        )

        check_performer = self._time_sync_helper.get_extra_data('check_performer', default=True)

        if check_performer:
            tag_entries = tag_entries.filter(performer__isnull=False)

        tag_entries = tag_entries.order_by('submitted_at')

        tags_to_bind = collection_to_mapping(
            tag_entries,
            key=(lambda t: (str(t.original_phone), t.performer_id if check_performer else None))
        )

        LOGGER.info('total tag entries to bind: {}'.format(sum(len(v) for v in tags_to_bind.values())))

        total_bound = 0

        for e in entries_to_bind:
            entry_staff_binding = e['staff_entry_binding__user_id']

            if check_performer:
                if entry_staff_binding is None:  # do not match entries without agent info
                    LOGGER.warning('entry with id {} has not staff entry binding as expected'.format(e['id']))
                    continue
            else:
                entry_staff_binding = None

            related_tags = tags_to_bind.get((str(e['phone']), entry_staff_binding), [])

            appropriate_tag = related_tags.pop(0) if related_tags else None

            if appropriate_tag is not None:
                appropriate_tag.request_id = e['id']
                appropriate_tag.save(update_fields=['request_id'])
                total_bound += 1

        LOGGER.info('successfully bound entry tag pairs: {}'.format(total_bound))

    def _get_entry_to_process_values(self, since, until):
        raise NotImplementedError


class RequestTagProcessingHelper(object):
    tag_description_helper = TagDescriptionHelper()

    def __init__(self):
        self._history_manager = RequestTagsHistoryManager()

    def upsert_request_tag(
            self, *, entry_id=None, tag_id, tag_type, request_origin, request_id=None,
            original_phone=None, original_user_id=None, related_phone=None, related_user_id=None,
            comment='', performed_by=None, **kwargs
    ):
        assert isinstance(request_origin, RequestOriginType)

        # note: original_user_trait is currently not used
        # original_user and related_user will be provided if corresponding values are not None;
        #   incorrect values have been checked beforehand

        entry_id = entry_id or uuid.uuid4()
        submitted_at = datetime_helper.utc_now()

        extra_fields = dict(
            submitted_at=submitted_at,
            performer=performed_by,
            tag_id=tag_id,
            tag_type=tag_type,
            request_id=request_id,
            request_origin=request_origin.value,
            original_phone=original_phone,
            original_user_id=original_user_id,
            related_phone=related_phone,
            related_user_id=related_user_id,
            comment=comment,
        )

        with transaction.atomic(savepoint=False):
            obj = RequestTagEntry.objects.filter(entry_id=entry_id).first()

            if obj is None:
                obj = RequestTagEntry.objects.create(entry_id=entry_id, **extra_fields)
                self._history_manager.add_entry(obj, operator_id=str(performed_by.id))
            else:
                for key, value in extra_fields.items():
                    setattr(obj, key, value)
                obj.save()
                self._history_manager.update_entry(obj, operator_id=str(performed_by.id))

        return obj

    def delete_request_tag(self, *, entry_id=None, tag_entry=None, performed_by=None, **kwargs):
        assert (entry_id is None) ^ (tag_entry is None), 'either tag_entry or entry_id must be specified'

        if tag_entry is None:
            tag_entry = RequestTagEntry.objects.get(entry_id=entry_id)

        with transaction.atomic(savepoint=False):
            self._history_manager.remove_entry(tag_entry, str(performed_by.id))
            tag_entry.delete()

    def get_all_request_origins(self):
        return [x.name for x in RequestOriginType]

    def get_all_tag_categories(self, request_origin, tag_type=None, enabled_only=True):
        assert isinstance(request_origin, RequestOriginType)
        assert isinstance(tag_type, RequestTagType)
        categories = self.tag_description_helper.get_all_tag_categories(
            tag_origin=request_origin.value, tag_type=tag_type.value, enabled_only=enabled_only
        )
        return categories
