import copy
import datetime
import logging

from django.db import transaction

from cars.core.util import datetime_helper

from cars.request_aggregator.models.call_center_common import CallStatSyncStatus, SyncOrigin

LOGGER = logging.getLogger(__name__)


class RequestTimeSyncHelper(object):
    def __init__(
            self, stat_sync_origin, use_timestamp=True, *,
            default_since,
            max_time_span=datetime.timedelta(hours=6),
            request_lag=datetime.timedelta(),
            request_overlap=datetime.timedelta()
    ):
        assert isinstance(stat_sync_origin, SyncOrigin)
        assert isinstance(default_since, datetime.datetime)
        assert all(isinstance(x, datetime.timedelta) for x in (max_time_span, request_lag, request_overlap))

        self._stat_sync_origin = stat_sync_origin.value
        self._use_timestamp = use_timestamp

        self._default_since = default_since
        self._max_time_span = max_time_span
        self._request_lag = request_lag
        self._request_overlap = request_overlap

    @property
    def sync_origin(self):
        return self._stat_sync_origin

    @property
    def max_time_span(self):
        return self._max_time_span

    @property
    def request_lag(self):
        return self._request_lag

    @property
    def request_overlap(self):
        return self._request_overlap

    def iter_time_span_to_process(self):
        sync_entry = self._get_sync_entry()

        if self.check_is_active(sync_entry):
            yield from self._iter_general_time_spans(sync_entry)
            yield from self._iter_extra_time_spans(sync_entry)

    def _get_sync_entry(self):
        with transaction.atomic(savepoint=False):
            sync_entry = CallStatSyncStatus.objects.filter(origin=self.sync_origin).first()

            # to be done: add unique constraint and integrity exception check
            if sync_entry is None:
                sync_entry = CallStatSyncStatus.objects.create(
                    origin=self.sync_origin, last_data_sync_time=self._default_since
                )

        return sync_entry

    def _get_sync_entry_for_update(self):
        sync_entry = CallStatSyncStatus.objects.select_for_update().get(origin=self.sync_origin)
        return sync_entry

    def pause(self, interval):
        with transaction.atomic(savepoint=False):
            sync_entry = self._get_sync_entry_for_update()
            sync_entry.active_since = datetime_helper.utc_now() + interval
            sync_entry.save()

    def stop(self):
        with transaction.atomic(savepoint=False):
            sync_entry = self._get_sync_entry_for_update()
            sync_entry.active_until = datetime_helper.utc_now()
            sync_entry.save()

    def check_is_active(self, sync_entry=None):
        if sync_entry is None:
            sync_entry = self._get_sync_entry()

        now = datetime_helper.utc_now()
        started = sync_entry.active_since is None or sync_entry.active_since <= now
        finished = sync_entry.active_until is not None and sync_entry.active_until < now

        is_active = started and not finished
        return is_active

    def _iter_general_time_spans(self, sync_entry):
        now = datetime_helper.utc_now()
        since = sync_entry.last_data_sync_time

        while since < now:
            since = min(now - self._request_lag, since - self._request_overlap)
            until = min(now, since + self._max_time_span)

            if self._use_timestamp:
                time_span = since.timestamp(), until.timestamp()
            else:
                time_span = since, until

            LOGGER.info('sync origin {}: processing time span {}'.format(self.sync_origin, time_span))

            yield time_span

            since = until

            self._update_last_data_sync_time(sync_entry, until)
            sync_entry.save()

    def _update_last_data_sync_time(self, sync_entry, until):
        sync_entry.last_data_sync_time = until

    def _iter_extra_time_spans(self, sync_entry):
        extra_time_spans = sync_entry.extra_time_spans

        while extra_time_spans:
            time_span = extra_time_spans.pop()

            if not self._use_timestamp:
                time_span = map(datetime_helper.timestamp_to_datetime, time_span)

            LOGGER.info('sync origin {}: processing time span {}'.format(self.sync_origin, time_span))

            yield time_span

            sync_entry.save()

    def add_extra_time_span(self, since, until=None):
        until = until or datetime_helper.timestamp_now()
        since, until = (x.timestamp() if isinstance(x, datetime.datetime) else x for x in (since, until))

        sync_entry = self._get_sync_entry()
        sync_entry.append_extra_time_span(since, until)
        sync_entry.save()

    def add_extra_data(self, key, value_to_append):
        with transaction.atomic(savepoint=False):
            sync_entry = CallStatSyncStatus.objects.select_for_update().get(origin=self.sync_origin)

            if sync_entry.data is None:
                sync_entry.data = {}

            sync_entry.data.setdefault(key, [])
            sync_entry.data[key].append(value_to_append)

            sync_entry.save()

    def remove_extra_data(self, key, value_to_remove):
        with transaction.atomic(savepoint=False):
            sync_entry = CallStatSyncStatus.objects.select_for_update().get(origin=self.sync_origin)

            if sync_entry.data and key in sync_entry.data:
                values = sync_entry.data[key]

                if value_to_remove in values:
                    values.remove(value_to_remove)

            sync_entry.save()

    def get_extra_data(self, key, default=None):
        sync_entry = CallStatSyncStatus.objects.get(origin=self.sync_origin)

        if sync_entry.data:
            return sync_entry.data.get(key, default)

        return default


class RequestDateSyncHelper(object):
    def __init__(self, stat_sync_origin, *, default_since):
        assert isinstance(stat_sync_origin, SyncOrigin)
        assert isinstance(default_since, datetime.date)

        self._stat_sync_origin = stat_sync_origin.value
        self._default_since = self._date_to_datetime(default_since)

    def _date_to_datetime(self, datetime_value):
        date_value = datetime_helper.utc_localize(datetime.datetime.combine(datetime_value, datetime.time.min))
        return date_value

    def _datetime_to_date(self, date_value):
        return date_value.date()

    @property
    def sync_origin(self):
        return self._stat_sync_origin

    def iter_dates_to_process(self, days_lag=0):
        assert isinstance(days_lag, int) and days_lag >= 0

        sync_entry = self._get_sync_entry()

        if self.check_is_active(sync_entry):
            yield from self._iter_general_dates(sync_entry, days_lag)
            yield from self._iter_extra_dates(sync_entry)

    def _get_sync_entry(self):
        with transaction.atomic(savepoint=False):
            sync_entry = CallStatSyncStatus.objects.filter(origin=self.sync_origin).first()

            # to be done: add unique constraint and integrity exception check
            if sync_entry is None:
                sync_entry = CallStatSyncStatus.objects.create(
                    origin=self.sync_origin, last_data_sync_time=self._default_since
                )

        return sync_entry

    def check_is_active(self, sync_entry):
        now = datetime_helper.utc_now()
        started = sync_entry.active_since is None or sync_entry.active_since <= now
        finished = sync_entry.active_until is not None and sync_entry.active_until < now
        return started and not finished

    def _iter_general_dates(self, sync_entry, days_lag):
        current_date = self._datetime_to_date(datetime_helper.utc_now())
        date_to_process = self._datetime_to_date(
            sync_entry.last_data_sync_time + datetime.timedelta(days=1) - datetime.timedelta(days=days_lag)
        )

        while date_to_process < current_date:
            LOGGER.info('sync origin {}: processing date {}'.format(self.sync_origin, date_to_process))

            yield date_to_process

            sync_entry.last_data_sync_time = max(sync_entry.last_data_sync_time, self._date_to_datetime(date_to_process))
            sync_entry.save()

            date_to_process += datetime.timedelta(days=1)

    def _iter_extra_dates(self, sync_entry):
        extra_time_spans = sync_entry.extra_time_spans

        while extra_time_spans:
            timestamp_value, _ = extra_time_spans.pop()
            date_value = self._datetime_to_date(datetime_helper.timestamp_to_datetime(timestamp_value))

            LOGGER.info('sync origin {}: processing date {}'.format(self.sync_origin, date_value))

            yield date_value

            sync_entry.save()

    def add_extra_date(self, date_value):
        datetime_value = self._date_to_datetime(date_value)
        since = until = datetime_value.timestamp()

        sync_entry = self._get_sync_entry()
        sync_entry.append_extra_time_span(since, until)
        sync_entry.save()


class RequestEntryCountSyncHelper(object):
    def __init__(self, stat_sync_origin, *, default_processed_count=0, max_entries_batch=1000):
        assert isinstance(stat_sync_origin, SyncOrigin)

        self._stat_sync_origin = stat_sync_origin.value
        self._default_processed_count = default_processed_count
        self._max_entries_batch = max_entries_batch

    @property
    def sync_origin(self):
        return self._stat_sync_origin

    def iter_indexes_to_process(self, total_count):
        sync_entry = self._get_sync_entry()

        if self.check_is_active(sync_entry):
            yield from self._iter_general_indexes(sync_entry, total_count)

    def _get_sync_entry(self):
        with transaction.atomic(savepoint=False):
            sync_entry = CallStatSyncStatus.objects.filter(origin=self.sync_origin).first()

            # to be done: add unique constraint and integrity exception check
            if sync_entry is None:
                sync_entry = CallStatSyncStatus.objects.create(
                    origin=self.sync_origin,
                    last_data_sync_time=datetime_helper.utc_now(),
                    data={'processed_count': self._default_processed_count},
                )

        return sync_entry

    def check_is_active(self, sync_entry):
        now = datetime_helper.utc_now()
        started = sync_entry.active_since is None or sync_entry.active_since <= now
        finished = sync_entry.active_until is not None and sync_entry.active_until < now
        return started and not finished

    def _iter_general_indexes(self, sync_entry, total_count):
        processed_count = sync_entry.data['processed_count']

        for offset in range(processed_count, total_count, self._max_entries_batch):
            end_offset = min(offset + self._max_entries_batch, total_count)

            data_range = offset, end_offset

            LOGGER.info('sync origin {}: processing data range {}'.format(self.sync_origin, data_range))

            yield data_range

            sync_entry.data['processed_count'] = end_offset
            sync_entry.last_data_sync_time = datetime_helper.utc_now()
            sync_entry.save()


class RequestStateSyncHelper(object):
    def __init__(self, stat_sync_origin, *, default_state=None):
        assert isinstance(stat_sync_origin, SyncOrigin)

        self._stat_sync_origin = stat_sync_origin.value
        self._default_state = default_state or {}

    @property
    def sync_origin(self):
        return self._stat_sync_origin

    def iter_states_to_process(self):
        sync_entry = self._get_sync_entry()

        if self.check_is_active(sync_entry):
            yield from self._iter_general_states(sync_entry)
            yield from self._iter_extra_states(sync_entry)

    def _get_sync_entry(self):
        with transaction.atomic(savepoint=False):
            sync_entry = CallStatSyncStatus.objects.filter(origin=self.sync_origin).first()

            # to be done: add unique constraint and integrity exception check
            if sync_entry is None:
                sync_entry = CallStatSyncStatus.objects.create(
                    origin=self.sync_origin,
                    last_data_sync_time=datetime_helper.utc_now(),
                    data=self._default_state,
                )

        return sync_entry

    def check_is_active(self, sync_entry):
        now = datetime_helper.utc_now()
        started = sync_entry.active_since is None or sync_entry.active_since <= now
        finished = sync_entry.active_until is not None and sync_entry.active_until < now
        return started and not finished

    def _iter_general_states(self, sync_entry):
        has_more = True

        while has_more:
            state = self._get_state(sync_entry)

            LOGGER.info('sync origin {}: processing state {}'.format(self.sync_origin, state))

            yield state

            has_more = self._update_state(sync_entry, state)

            sync_entry.last_data_sync_time = datetime_helper.utc_now()
            sync_entry.save()

    def _get_state(self, sync_entry):
        return copy.deepcopy(sync_entry.data)

    def _update_state(self, sync_entry, state=None):
        if state is not None:
            sync_entry.data = state

        has_more = False
        return has_more

    def _iter_extra_states(self, sync_entry):
        return ()
