import collections
import concurrent.futures
import logging
import os

from cars.settings import REQUEST_AGGREGATOR as settings

from .base import StatisticsHelperBase
from .export import SolomonExportHelper, YtExportHelper, LogExportHelper

LOGGER = logging.getLogger(__name__)


class AggregatedStatisticsHelper(object):
    MAX_WORKERS_COUNT = (os.cpu_count() or 1) * 5

    def __init__(self, yt_export_table=None):
        self._helpers = {}
        self._exporters = self._init_exporters(yt_export_table)

    @classmethod
    def from_settings(cls):
        return cls(
            yt_export_table=settings['monitoring']['export']['table']
        )

    def _init_exporters(self, yt_export_table=None):
        exporters = {
            'solomon': SolomonExportHelper('monitoring'),
            'logging': LogExportHelper(),
        }
        if yt_export_table is not None:
            exporters['yt'] = YtExportHelper(yt_export_table)
        return exporters

    def register(self, name):
        def wrapper(helper_cls):
            assert issubclass(helper_cls, StatisticsHelperBase)
            self._helpers[name] = helper_cls()

        return wrapper

    def report_stats(self):
        flattened_statistics = self._collect_stats()
        self._run_async((e.export, (flattened_statistics, ), {}) for e in self._exporters.values())

    def _collect_stats(self):
        statistics = {k: v.collect() for k, v in self._helpers.items()}
        flattened_statistics = self._flatten_data(statistics)
        return flattened_statistics

    def _flatten_data(self, mapping, *, prefix=None):
        assert isinstance(mapping, collections.Mapping)

        flattened_data = {}

        for k, v in mapping.items():
            flattened_key = '{}.{}'.format(prefix, k) if prefix is not None else k

            if isinstance(v, collections.Mapping):
                nested_values = self._flatten_data(v, prefix=flattened_key)
                flattened_data.update(nested_values)
            else:
                flattened_data[flattened_key] = v

        return flattened_data

    def optimize_stats(self):
        self._run_async((e.optimize, (), {}) for e in self._exporters.values())

    def _run_async(self, functor_collection):
        with concurrent.futures.ThreadPoolExecutor(self.MAX_WORKERS_COUNT) as executor:
            futures = (executor.submit(functor, *args, **kwargs) for functor, args, kwargs in functor_collection)

            for future in concurrent.futures.as_completed(futures):
                try:
                    future.result()
                except Exception as exc:
                    LOGGER.exception('future raised an exception: {}'.format(exc))
                    raise


statistics_helper = AggregatedStatisticsHelper.from_settings()
register = statistics_helper.register
