import collections
import statistics


class StatisticsHelperBase(object):
    @classmethod
    def update_collection(cls, base, update):
        for k, v in update.items():
            if isinstance(v, collections.Mapping):
                base[k] = cls.update_collection(base.get(k, {}), v)
            elif isinstance(v, collections.Sequence):
                base[k] = base.get(k, type(v)()) + v
            else:
                base[k] = base.get(k, 0) + v
        return base

    def get_default_queue_stat_collection(self):
        stats = {
            # for started calls
            'total': 0,
            'pending': 0,
            'serviced': 0,
            # for finished calls
            'total_managed': 0,
            'not_serviced': 0,
            'completed': 0,
            'durations': [],
        }
        return stats

    def collect(self):
        """
        :return: collected statistics
        """
        raise NotImplementedError

    def get_active_operators(self, since, until):
        return None

    def get_call_entries_stat(self, since, until):
        """
        :return: mapping of queue stats
        """
        return None

    def annotate_call_duration(self, stat):
        for q, q_stat in stat.items():
            durations = q_stat['durations']

            q_stat['min_duration'] = min(durations, default=0)
            q_stat['max_duration'] = max(durations, default=0)
            q_stat['mean_duration'] = statistics.mean(durations) if durations else 0
            q_stat['median_duration'] = statistics.median(durations) if durations else 0


class ExportHelperBase(object):
    def export(self, flattened_statistics):
        raise NotImplementedError

    def optimize(self):
        pass
