import datetime
import logging

from django.db import transaction
from django.db.models import Max

import cars.settings
from cars.core.util import datetime_helper

from cars.request_aggregator.models.audiotele_stats import (
    AudioteleIncomingCallEntry, AudioteleCallDirection, AudioteleCallAction
)
from cars.request_aggregator.models.chat2desk_stats import Chat2DeskClientEntry
from cars.request_aggregator.models.internal_cc_stats import (
    CallCenterEntry, InternalCCVerb, CallCenterAltayOutgoingEntry
)
from cars.request_aggregator.models.call_center_common import CallStatSyncStatus, SyncOrigin
from cars.request_aggregator.models.call_tags import RequestTagEntry, TagOrigin

from cars.users.models import User

from .base import StatisticsHelperBase
from .aggregated import register


LOGGER = logging.getLogger(__name__)


@register('matching')
class RequestMatchingStatisticsHelper(StatisticsHelperBase):
    ENTRIES_MATCHING_BATCH = 5000

    def collect(self):
        stats = {}
        self.update_collection(stats, self._get_internal_cc_stats())
        self.update_collection(stats, self._get_audiotele_stats())
        self.update_collection(stats, self._get_chat2desk_stats())
        return stats

    def _get_internal_cc_stats(self):
        internal_cc_stats = {}

        origin = SyncOrigin.CC_INTERNAL_USERS_BINDING

        sync_entry = self._get_sync_entry(origin)

        while True:
            last_synced_id = sync_entry.data['last_synced_id']
            last_synced_id_upper = min(
                last_synced_id + self.ENTRIES_MATCHING_BATCH,
                CallCenterEntry.objects.aggregate(Max('id'))['id__max']
            )
            entries_to_process = list(
                CallCenterEntry.objects.filter(
                    id__gt=last_synced_id, id__lte=last_synced_id_upper,
                    verb=InternalCCVerb.ENTER_QUEUE.value, phone__isnull=False,
                )
                .values('id', 'phone')
            )

            self._update_sync_entry_data(sync_entry, entries_to_process, last_synced_id_upper)

            if not entries_to_process:
                break

        internal_cc_stats['cc_internal_matched_users_ratio'] = self._calculate_ratio(sync_entry.data)
        return internal_cc_stats

    def _get_audiotele_stats(self):
        audiotele_stats = {}

        origin = SyncOrigin.CC_AUDIOTELE_USERS_BINDING

        sync_entry = self._get_sync_entry(origin)

        while True:
            last_synced_id = sync_entry.data['last_synced_id']
            last_synced_id_upper = min(
                last_synced_id + self.ENTRIES_MATCHING_BATCH,
                AudioteleIncomingCallEntry.objects.aggregate(Max('id'))['id__max']
            )
            entries_to_process = list(
                AudioteleIncomingCallEntry.objects.filter(
                    id__gt=last_synced_id, id__lte=last_synced_id_upper,
                    direction=AudioteleCallDirection.INCOMING.value,
                    action=AudioteleCallAction.FINISH.value, phone__isnull=False
                )
                .values('id', 'phone')
            )

            self._update_sync_entry_data(sync_entry, entries_to_process, last_synced_id_upper)

            if not entries_to_process:
                break

        audiotele_stats['cc_audiotele_matched_users_ratio'] = self._calculate_ratio(sync_entry.data)
        return audiotele_stats

    def _get_chat2desk_stats(self):
        chat2desk_stats = {}

        origin = SyncOrigin.CHAT2DESK_USERS_BINDING

        sync_entry = self._get_sync_entry(origin)

        while True:
            last_synced_id = sync_entry.data['last_synced_id']
            last_synced_id_upper = min(
                last_synced_id + self.ENTRIES_MATCHING_BATCH,
                Chat2DeskClientEntry.objects.aggregate(Max('id'))['id__max']
            )
            entries_to_process = list(
                Chat2DeskClientEntry.objects.filter(
                    id__gt=last_synced_id, id__lte=last_synced_id_upper,
                )
                .values('id', 'assigned_phone')
            )

            self._update_sync_entry_data(sync_entry, entries_to_process, last_synced_id_upper, entry_phone_field='assigned_phone')

            if not entries_to_process:
                break

        chat2desk_stats['chat2desk_matched_users_ratio'] = self._calculate_ratio(sync_entry.data)
        return chat2desk_stats

    def _get_sync_entry(self, origin):
        with transaction.atomic(savepoint=False):
            sync_entry = CallStatSyncStatus.objects.filter(origin=origin.value).first()

            # to be done: add unique constraint and integrity exception check
            if sync_entry is None:
                sync_entry = CallStatSyncStatus.objects.create(
                    origin=origin.value,
                    last_data_sync_time=datetime_helper.utc_now(),
                    data={'last_synced_id': 0, 'processed': 0, 'matched': 0}
                )

        return sync_entry

    def _update_sync_entry_data(self, sync_entry, entries_to_process, last_synced_id, *, entry_phone_field='phone'):
        sync_entry.data['last_synced_id'] = last_synced_id

        if entries_to_process:
            phones_to_process = {e[entry_phone_field] for e in entries_to_process if e[entry_phone_field]}

            matched_phones = set(User.objects.filter(phone__in=phones_to_process).values_list('phone', flat=True))

            matched_entries_count = sum(1 for e in entries_to_process if e[entry_phone_field] in matched_phones)

            sync_entry.data['processed'] += len(entries_to_process)
            sync_entry.data['matched'] += matched_entries_count

        sync_entry.last_data_sync_time = datetime_helper.utc_now()
        sync_entry.save()

    def _calculate_ratio(self, data):
        return data['matched'] / (data['processed'] or 1)


@register('tag_matching')
class RequestTagMatchingStatisticsHelper(StatisticsHelperBase):
    WINDOW_SIZE = datetime.timedelta(days=30)
    TICK_PERIOD = datetime.timedelta(minutes=15)

    def __init__(self):
        super().__init__()
        self._next_sync_time = datetime_helper.utc_now()

    def _check_need_update(self):
        now = datetime_helper.utc_now()
        need_update = (self._next_sync_time <= now)

        if need_update:
            self._next_sync_time = now + self.TICK_PERIOD

        return need_update

    def collect(self):
        stats = {}

        if self._check_need_update():
            self.update_collection(stats, self._get_cc_internal_incoming_stats())
            self.update_collection(stats, self._get_cc_internal_outgoing_stats())
            self.update_collection(stats, self._get_audiotele_cc_incoming_stats())
            self.update_collection(stats, self._get_audiotele_cc_outgoing_stats())

        return stats

    def _get_cc_internal_incoming_stats(self):
        since = datetime_helper.utc_now() - self.WINDOW_SIZE

        tag_origins = (TagOrigin.CARSHARING.value, TagOrigin.CARSHARING_VIP.value)

        total_tags_count = self._get_total_tags_count(since, tag_origins)
        bound_tags_count = self._get_bound_tags_count(since, tag_origins)

        total_call_count = (
            CallCenterEntry.objects.using(cars.settings.DB_RO_ID)
            .filter(
                time_id__gte=since,
                verb=InternalCCVerb.CONNECT.value,
            )
            .count()
        )

        stats = self._fill_stats('cc_internal.incoming', total_call_count, total_tags_count, bound_tags_count)
        return stats

    def _get_cc_internal_outgoing_stats(self):
        since = datetime_helper.utc_now() - self.WINDOW_SIZE

        tag_origins = (TagOrigin.CC_INTERNAL_ALTAY_OUTGOING.value, )

        total_tags_count = self._get_total_tags_count(since, tag_origins)
        bound_tags_count = self._get_bound_tags_count(since, tag_origins)

        total_call_count = (
            CallCenterAltayOutgoingEntry.objects.using(cars.settings.DB_RO_ID)
            .filter(
                time_enter__gte=since,
            )
            .count()
        )

        stats = self._fill_stats('cc_internal.outgoing', total_call_count, total_tags_count, bound_tags_count)
        return stats

    def _get_audiotele_cc_incoming_stats(self):
        since = datetime_helper.utc_now() - self.WINDOW_SIZE

        tag_origins = (TagOrigin.AUDIOTELE_INCOMING.value, )

        total_tags_count = self._get_total_tags_count(since, tag_origins)
        bound_tags_count = self._get_bound_tags_count(since, tag_origins)

        total_call_count = (
            AudioteleIncomingCallEntry.objects.using(cars.settings.DB_RO_ID)
            .filter(
                time_enter__gte=since,
                action=AudioteleCallAction.FINISH.value,
                direction=AudioteleCallDirection.INCOMING.value,
                is_answered=True,
            )
            .count()
        )

        stats = self._fill_stats('cc_audiotele.incoming', total_call_count, total_tags_count, bound_tags_count)
        return stats

    def _get_audiotele_cc_outgoing_stats(self):
        since = datetime_helper.utc_now() - self.WINDOW_SIZE

        tag_origins = (TagOrigin.AUDIOTELE_OUTGOING.value, )

        total_tags_count = self._get_total_tags_count(since, tag_origins)
        bound_tags_count = self._get_bound_tags_count(since, tag_origins)

        total_call_count = (
            AudioteleIncomingCallEntry.objects.using(cars.settings.DB_RO_ID)
            .filter(
                time_enter__gte=since,
                action=AudioteleCallAction.FINISH.value,
                direction=AudioteleCallDirection.OUTGOING.value,
            )
            .count()
        )

        stats = self._fill_stats('cc_audiotele.ougoing', total_call_count, total_tags_count, bound_tags_count)
        return stats

    def _fill_stats(self, prefix, total_call_count, total_tags_count, bound_tags_count):
        stats = {}

        suffix = '{}d'.format(self.WINDOW_SIZE.days)

        stats['total_call_count'] = total_call_count
        stats['total_tags_count'] = total_tags_count
        stats['bound_tags_count'] = bound_tags_count
        stats['bound_tags_ratio'] = bound_tags_count / (total_tags_count or 1.0)
        stats['matched_calls_ratio'] = bound_tags_count / (total_call_count or 1.0)

        stats = {'.'.join((prefix, k, suffix)): v for k, v in stats.items()}
        return stats

    def _get_total_tags_count(self, since, origins):
        total_tags_count = (
            RequestTagEntry.objects.using(cars.settings.DB_RO_ID)
            .filter(
                submitted_at__gte=since,
                request_origin__in=origins,
            )
            .count()
        )
        return total_tags_count

    def _get_bound_tags_count(self, since, origins):
        bound_tag_request_ids = (
            RequestTagEntry.objects.using(cars.settings.DB_RO_ID)
            .filter(
                submitted_at__gte=since,
                request_origin__in=origins,
                request_id__isnull=False,
            )
            .values_list('request_id', flat=True)
        )
        bound_tags_count = len(set(bound_tag_request_ids))
        return bound_tags_count
