import logging

from cars.core.daemons import CarsharingDaemon
from cars.core.util import handle_error

from cars.request_aggregator.core import (
    # internal
    IncomingTelephonyCollectingHelper,
    TelephonyAssignmentCollectingHelper,
    OutgoingTelephonyCollectingHelper,
    CCInternalAltayOutCollectingHelper,

    # audiotele
    AudioteleCallStatCollectingHelper,
    AudioteleCallTrackCollectingHelper,
    AudioteleCallTrackBindingHelper,
    # chat2desk
    Chat2DeskCollectingHelper,
)

from .base import CollectingDaemonBase


LOGGER = logging.getLogger(__name__)


class TelStatCollectingDaemon(CollectingDaemonBase):
    tick_interval = '* * * * * */30'  # 30 sec

    def __init__(self, *args, **kwargs):
        super().__init__(helper=IncomingTelephonyCollectingHelper.from_settings(), *args, **kwargs)

    def get_daemon_name(self):
        return 'tel_stat_collector'


class OutgoingTelStatCollectingDaemon(CollectingDaemonBase):
    tick_interval = '* * * * * */30'  # 30 sec

    def __init__(self, *args, **kwargs):
        super().__init__(helper=OutgoingTelephonyCollectingHelper.from_settings(), *args, **kwargs)

    def get_daemon_name(self):
        return 'out_tel_stat_collector'


class OutgoingAltayStatCollectingDaemon(CollectingDaemonBase):
    tick_interval = '*/2 * * * *'  # 2 min

    def __init__(self, *args, **kwargs):
        super().__init__(helper=CCInternalAltayOutCollectingHelper.from_settings(), *args, **kwargs)

    def get_daemon_name(self):
        return 'altay_out_stat_collector'


class TelephonyAssignmentUpdatingDaemon(CollectingDaemonBase):
    tick_interval = '* * * * * *'  # 1 sec (as fast as it can)

    def __init__(self, *args, **kwargs):
        super().__init__(helper=TelephonyAssignmentCollectingHelper.from_settings(), *args, **kwargs)

    def get_daemon_name(self):
        return 'tel_assignment_updater'

    def _do_tick(self):
        with handle_error(re_raise=False):
            hung_entries_count = self._helper.process_hung_entries()
            if hung_entries_count:
                self._report_tick_processed_entries_count({'hung': hung_entries_count})

        super()._do_tick()


class AudioteleCallSyncDaemon(CarsharingDaemon):

    tick_interval = '0 3-18 * * *'  # every hour starting from 3 utc

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._sync_helper = AudioteleCallStatCollectingHelper.from_settings()
        self._track_downloading_helper = AudioteleCallTrackCollectingHelper.from_settings()
        self._track_binding_helper = AudioteleCallTrackBindingHelper.from_settings()

    def get_distributed_lock_relative_path(self):
        return 'request_aggregator/locks/audiotele_call_sync_daemon.lock'

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.audiotele_call_sync_daemon'

    def get_solomon_service(self):
        return 'request_aggregator'

    def _do_tick(self):
        processed_entries_count = 0

        try:
            processed_entries_count = self._sync_helper.update_data()
        finally:
            self._report_tick_processed_entries_count(processed_entries_count)

        with handle_error(re_raise=False):
            self._track_downloading_helper.update_data()

        with handle_error(re_raise=False):
            self._track_binding_helper.update_track_bindings()

    def _report_tick_processed_entries_count(self, processed_entries_count):
        sensor_name = '.'.join((self.get_solomon_sensor_prefix(), 'tick.processed_entries_count'))
        try:
            self.solomon.set_value(sensor_name, processed_entries_count)
        except Exception as exc:
            LOGGER.error('error reporting value of sensor "{}"'.format(sensor_name))


class Chat2DeskCollectingDaemon(CarsharingDaemon):

    tick_interval = '*/15 * * * *'  # 15 minutes

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._chat2desk_helper = Chat2DeskCollectingHelper.from_settings()

    def get_distributed_lock_relative_path(self):
        return 'request_aggregator/locks/chat2desk_collector.lock'

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.chat2desk_collector'

    def get_solomon_service(self):
        return 'request_aggregator'

    def _do_tick(self):
        processed_entries_counts = ()

        try:
            processed_entries_counts = self._chat2desk_helper.update_data()
        finally:
            processed_entries_count = sum(processed_entries_counts)
            self._report_tick_processed_entries_count(processed_entries_count)

    def _report_tick_processed_entries_count(self, processed_entries_count):
        sensor_name = '.'.join((self.get_solomon_sensor_prefix(), 'tick.processed_entries_count'))
        try:
            self.solomon.set_value(sensor_name, processed_entries_count)
        except Exception as exc:
            LOGGER.error('error reporting value of sensor "{}"'.format(sensor_name))
