import logging

import croniter

from cars.core.daemons import CarsharingDaemon, CarsharingWorker
from cars.core.util import datetime_helper, handle_error

from cars.settings import REQUEST_AGGREGATOR as settings

from cars.request_aggregator.core import OperatorsInfoHelper, YangWorkersInfoHelper, TraitExportHelper
from cars.request_aggregator.core.chat2desk import Chat2deskExportHelper
from cars.request_aggregator.core.request_tags import RequestTagExportingHelper

from cars.request_aggregator.core.internal_cc.incoming.exporting_helper import InternalCCIncomingDataExportingHelper
from cars.request_aggregator.core.internal_cc.routing.exporting_helper import InternalCCRoutingDataExportingHelper
from cars.request_aggregator.core.internal_cc.outgoing.exporting_helper import InternalCCOutgoingDataExportingHelper
from cars.request_aggregator.core.audiotele.exporting_helper import (
    AudioteleCCDataExportingHelper,
    AudioteleCCTrackDataExportingHelper
)

from cars.request_aggregator.models.call_center_common import SyncOrigin
from cars.request_aggregator.models.call_tags import TagOrigin

LOGGER = logging.getLogger(__name__)


class OperatorsExportingDaemon(CarsharingDaemon):
    def get_distributed_lock_relative_path(self):
        return 'request_aggregator/locks/operators_exporter.lock'

    def get_solomon_service(self):
        return 'request_aggregator'

    get_solomon_sensor_prefix = None
    _do_tick = None

    def init_workers(self):
        self.register_worker('operators_info_updater', OperatorsInfoUpdatingWorker)

        self.register_worker('operators_info_export', OperatorsInfoExportingWorker)

        self.register_worker('roles_export', RolesExportingWorker)

        self.register_worker('chats_export', Chat2deskExportingWorker)

        # call data exporters
        self.register_worker('internal_cc_incoming', InternalCCIncomingCallDataExportingWorker)

        self.register_worker('internal_cc_routing', InternalCCRoutingDataExportingWorker)

        self.register_worker('internal_cc_outgoing', InternalCCOutgoingCallDataExportingWorker)

        self.register_worker('audiotele_cc', AudioteleCCCallDataExportingWorker)

        self.register_worker('audiotele_cc_tracks', AudioteleCCCallTrackDataExportingWorker)

        # common tag exporter
        self.register_worker(
            'tag_export_common', TagExportingWorker.make_custom(
                tag_origin=None,
                sync_origin=SyncOrigin.REQUEST_TAGS_EXPORTING,
                export_table_path=settings['tags']['export']['common_table'],
                add_request_origin=True,
            )
        )

        # partial (specific) tag exporters
        self.register_worker(
            'tag_export_cc_internal_carsharing', TagExportingWorker.make_custom(
                tag_origin=TagOrigin.CARSHARING,
                sync_origin=SyncOrigin.CC_INTERNAL_CARSHARING_TAGS_EXPORTING,
                export_table_path=settings['tags']['export']['cc_internal_carsharing_table'],
            )
        )
        self.register_worker(
            'tag_export_cc_internal_carsharing_vip', TagExportingWorker.make_custom(
                tag_origin=TagOrigin.CARSHARING_VIP,
                sync_origin=SyncOrigin.CC_INTERNAL_CARSHARING_VIP_TAGS_EXPORTING,
                export_table_path=settings['tags']['export']['cc_internal_carsharing_vip_table'],
            )
        )
        self.register_worker(
            'tag_export_cc_internal_altay_outgoing', TagExportingWorker.make_custom(
                tag_origin=TagOrigin.CC_INTERNAL_ALTAY_OUTGOING,
                sync_origin=SyncOrigin.CC_INTERNAL_ALTAY_OUTGOING_TAGS_EXPORTING,
                export_table_path=settings['tags']['export']['cc_internal_altay_outgoing_table'],
            )
        )
        self.register_worker(
            'tag_export_cc_audiotele_incoming', TagExportingWorker.make_custom(
                tag_origin=TagOrigin.AUDIOTELE_INCOMING,
                sync_origin=SyncOrigin.CC_AUDIOTELE_INCOMING_TAGS_EXPORTING,
                export_table_path=settings['tags']['export']['cc_audiotele_incoming_table'],
            )
        )
        self.register_worker(
            'tag_export_cc_audiotele_outgoing', TagExportingWorker.make_custom(
                tag_origin=TagOrigin.AUDIOTELE_OUTGOING,
                sync_origin=SyncOrigin.CC_AUDIOTELE_OUTGOING_TAGS_EXPORTING,
                export_table_path=settings['tags']['export']['cc_audiotele_outgoing_table'],
            )
        )


class OperatorsInfoUpdatingWorker(CarsharingWorker):
    tick_interval = '15 * * * *'  # hourly at xx:15
    re_raise = False

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._operators_info_helper = OperatorsInfoHelper.from_settings()

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.operators_updater'

    def _do_tick(self):
        self._operators_info_helper.update_staff_entries()


class OperatorsInfoExportingWorker(CarsharingWorker):
    tick_interval = '0 5,17 * * *'  # twice a day
    re_raise = False

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._operators_info_helper = OperatorsInfoHelper.from_settings()
        self._yang_workers_info_helper = YangWorkersInfoHelper.from_settings()

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.operators_exporter'

    def _do_tick(self):
        with handle_error(re_raise=False):
            self._yang_workers_info_helper.update_staff_entries()

        self._operators_info_helper.export_staff_entries(check_data_exists=True)


class RolesExportingWorker(CarsharingWorker):
    tick_interval = '0 */4 * * *'  # every 4 hours

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._white_listed_roles_export_helper = TraitExportHelper.from_cc_included_roles_settings()
        self._black_listed_roles_export_helper = TraitExportHelper.from_cc_excluded_roles_settings()
        self._user_available_info_roles_export_helper = TraitExportHelper.from_cc_user_available_info_settings()

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.roles_exporter'

    def _do_tick(self):
        with handle_error(re_raise=False):
            self._white_listed_roles_export_helper.export_user_roles(check_data_exists=True)

        with handle_error(re_raise=False):
            self._black_listed_roles_export_helper.export_user_roles(check_data_exists=True)

        with handle_error(re_raise=False):
            self._user_available_info_roles_export_helper.export_user_roles(check_data_exists=False, append=False)


class Chat2deskExportingWorker(CarsharingWorker):
    tick_interval = '5 * * * *'  # every hour at x:05
    optimization_tick_interval = '0 */12 * * *'  # every 12 hours

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._export_helper = Chat2deskExportHelper.from_settings()

        self._optimization_cron = croniter.croniter(self.optimization_tick_interval)
        self._next_optimization_time = datetime_helper.utc_localize(
            self.get_next_run_time(self._optimization_cron)
        )

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.chat2desk_exporter'

    def _do_tick(self):
        self._export_helper.export()

        if datetime_helper.utc_now() >= self._next_optimization_time:
            self._export_helper.optimize()
            self._next_optimization_time = datetime_helper.utc_localize(
                self.get_next_run_time(self._optimization_cron)
            )


class TagExportingWorker(CarsharingWorker):
    tick_interval = '*/5 * * * *'  # every 5 minutes
    optimization_tick_interval = '0 */6 * * *'  # every 6 hours

    _worker_helper_kwargs = {}

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._helper = RequestTagExportingHelper(**self._worker_helper_kwargs)

        self._optimization_cron = croniter.croniter(self.optimization_tick_interval)
        self._next_optimization_time = datetime_helper.utc_localize(
            self.get_next_run_time(self._optimization_cron)
        )

    @classmethod
    def make_custom(cls, *, sync_origin, tag_origin, export_table_path, add_request_origin=False):
        assert isinstance(sync_origin, SyncOrigin)
        assert tag_origin is None or isinstance(tag_origin, TagOrigin)

        class CustomTagExportingWorker(cls):
            _worker_helper_kwargs = {
                'sync_origin': sync_origin,
                'tag_origin': tag_origin,
                'export_table_path': export_table_path,
                'add_request_origin': add_request_origin,
            }

        return CustomTagExportingWorker

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.request_tags_exporter'

    def _do_tick(self):
        self._helper.perform()

        if datetime_helper.utc_now() >= self._next_optimization_time:
            self._helper.optimize()
            self._next_optimization_time = datetime_helper.utc_localize(
                self.get_next_run_time(self._optimization_cron)
            )


class CallDataExportingWorkerBase(CarsharingWorker):
    tick_interval = '*/5 * * * *'  # every 5 minutes
    optimization_tick_interval = '0 */6 * * *'  # every 6 hours

    _helper = None

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self._optimization_cron = croniter.croniter(self.optimization_tick_interval)
        self._next_optimization_time = datetime_helper.utc_localize(
            self.get_next_run_time(self._optimization_cron)
        )

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.call_data_exporter'

    def _do_tick(self):
        self._helper.perform()

        if datetime_helper.utc_now() >= self._next_optimization_time:
            self._helper.optimize()
            self._next_optimization_time = datetime_helper.utc_localize(
                self.get_next_run_time(self._optimization_cron)
            )


class InternalCCIncomingCallDataExportingWorker(CallDataExportingWorkerBase):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._helper = InternalCCIncomingDataExportingHelper.make(
            SyncOrigin.CC_INTERNAL_INCOMING_DATA_EXPORTING,
            settings['call_data']['export']['cc_internal_incoming_table']
        )


class InternalCCOutgoingCallDataExportingWorker(CallDataExportingWorkerBase):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._helper = InternalCCOutgoingDataExportingHelper.make(
            SyncOrigin.CC_INTERNAL_OUTGOING_DATA_EXPORTING,
            settings['call_data']['export']['cc_internal_outgoing_table']
        )


class InternalCCRoutingDataExportingWorker(CallDataExportingWorkerBase):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._helper = InternalCCRoutingDataExportingHelper.make(
            SyncOrigin.CC_INTERNAL_ROUTING_DATA_EXPORTING,
            settings['call_data']['export']['cc_internal_routing_table']
        )


class AudioteleCCCallDataExportingWorker(CallDataExportingWorkerBase):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._helper = AudioteleCCDataExportingHelper.make(
            SyncOrigin.CC_AUDIOTELE_DATA_EXPORTING,
            settings['call_data']['export']['cc_audiotele_table']
        )


class AudioteleCCCallTrackDataExportingWorker(CallDataExportingWorkerBase):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._helper = AudioteleCCTrackDataExportingHelper.make(
            SyncOrigin.CC_AUDIOTELE_TRACK_DATA_EXPORTING,
            settings['call_data']['export']['cc_audiotele_track_table']
        )
