import logging

from cars.core.daemons import CarsharingDaemon, CarsharingWorker

from cars.request_aggregator.core.request_tags.collecting_helper import RequestTagCollectingHelper
from cars.request_aggregator.core.request_tags.collecting_helper_yql import RequestTagYQLCollectingHelper
from cars.request_aggregator.models import TagOrigin

LOGGER = logging.getLogger(__name__)


class RequestTagsUpdatingDaemon(CarsharingDaemon):
    def get_distributed_lock_relative_path(self):
        return 'request_aggregator/locks/request_tags_updater.lock'

    def get_solomon_service(self):
        return 'request_aggregator'

    _do_tick = None
    get_solomon_sensor_prefix = None

    def init_workers(self):
        for tag_origin in TagOrigin:
            if not tag_origin.is_real_time:
                self.register_worker(tag_origin.name, RequestTagsUpdatingWorker.make_custom(tag_origin))

        self.register_worker('common_yql_updater', RequestTagsYQLUpdatingWorker)


class RequestTagsUpdatingWorker(CarsharingWorker):
    tick_interval = '*/5 * * * *'  # 5 min

    tag_origin = None

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._tags_helper = RequestTagCollectingHelper(self.tag_origin)

    @classmethod
    def make_custom(cls, tag_origin_):
        assert isinstance(tag_origin_, TagOrigin)

        class CustomTagExportingWorker(cls):
            tag_origin = tag_origin_

        return CustomTagExportingWorker

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.request_tags_updater.{}'.format(self.tag_origin.name)

    def _do_tick(self):
        processed_entries_count = self._tags_helper.update_data()
        self._report_tick_processed_entries_count(processed_entries_count)

    def _report_tick_processed_entries_count(self, processed_entries_count):
        sensor_name = '.'.join((self.get_solomon_sensor_prefix(), 'tick.processed_entries_count'))
        try:
            self.solomon.set_value(sensor_name, processed_entries_count)
        except Exception as exc:
            LOGGER.error('error reporting value of sensor "{}"'.format(sensor_name))


class RequestTagsYQLUpdatingWorker(CarsharingWorker):
    tick_interval = '30 * * * *'  # every 30th minute once an hour

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._tags_helper = RequestTagYQLCollectingHelper.from_settings()

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.request_tags_updater.common_yql_updater'

    def _do_tick(self):
        entries_stat = self._tags_helper.update_data()
        self._report_tick_processed_entries_count(entries_stat)

    def _report_tick_processed_entries_count(self, entries_stat):
        for name, value in entries_stat.items():
            full_name = '.'.join((self.get_solomon_sensor_prefix(), name))
            try:
                self.solomon.set_value(full_name, value)
            except Exception as exc:
                LOGGER.exception('error reporting value of sensor "{}"'.format(full_name))
