import logging

import croniter

from cars.core.util import datetime_helper
from cars.request_aggregator.core.internal_cc.load_balance_sync_helper import LoadBalanceSyncHelper
from cars.request_aggregator.core.statistics import statistics_helper

# register monitors
from cars.request_aggregator.core.statistics import request_matching
from cars.request_aggregator.core.audiotele import stat_helper
from cars.request_aggregator.core.internal_cc import stat_helper

from cars.core.daemons import CarsharingDaemon, CarsharingWorker


LOGGER = logging.getLogger(__name__)


class TelStatMonitoringDaemon(CarsharingDaemon):
    def get_distributed_lock_relative_path(self):
        return 'request_aggregator/locks/monitoring.lock'

    def get_solomon_service(self):
        return 'request_aggregator'

    get_solomon_sensor_prefix = None
    _do_tick = None

    def init_workers(self):
        self.register_worker('monitoring', TelStatMonitoringWorker)
        self.register_worker('load_balance_watcher', CCLoadBalanceSettingsWorker)


class TelStatMonitoringWorker(CarsharingWorker):
    tick_interval = '* * * * *'  # every 1 minute
    optimization_tick_interval = '0 * * * *'  # every hour

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._helper = statistics_helper

        self._optimization_cron = croniter.croniter(self.optimization_tick_interval)
        self._next_optimization_time = datetime_helper.utc_localize(
            self.get_next_run_time(self._optimization_cron)
        )

    def get_solomon_sensor_prefix(self):
        return 'request_aggregator.monitoring'

    def _do_tick(self):
        self._helper.report_stats()

        if datetime_helper.utc_now() >= self._next_optimization_time:
            self._helper.optimize_stats()
            self._next_optimization_time = datetime_helper.utc_localize(
                self.get_next_run_time(self._optimization_cron)
            )


class CCLoadBalanceSettingsWorker(CarsharingWorker):
    tick_interval = '* * * * *'  # every 1 minute

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._helper = LoadBalanceSyncHelper.from_settings()

    def get_solomon_sensor_prefix(self):
        return 'callcenter.load_balance'

    def _do_tick(self):
        self._helper.process()
