import datetime
import enum
import os
import uuid

from django.db import models
from django.contrib.postgres.fields import JSONField
from phonenumber_field.modelfields import PhoneNumberField

from cars.core.util import datetime_helper, phone_number_helper
from cars.callcenter.models import CallCenterStaffEntry
from .request_user_phone_binding import RequestUserPhoneBindingEntry


class AudioteleCallDirection(enum.Enum):
    INCOMING = 'i'
    OUTGOING = 'o'


class AudioteleCallAction(enum.Enum):
    START = 'start'
    FINISH = 'finish'


class AudioteleIncomingCallEntry(models.Model):
    direction = models.CharField(
        max_length=16,
        choices=[(x.value, x.name) for x in AudioteleCallDirection],
    )

    action = models.CharField(
        max_length=16,
        choices=[(x.value, x.name) for x in AudioteleCallAction],
    )

    related_call_id = models.CharField(  # audiotele internal call id
        max_length=64,
    )

    time_enter = models.DateTimeField()

    time_connect = models.DateTimeField(null=True)  # not answered call

    time_exit = models.DateTimeField(null=True)  # active call

    is_answered = models.BooleanField(default=False)

    duration = models.PositiveIntegerField(null=True)  # active call, not answered call

    phone = PhoneNumberField(null=True)

    # agent username
    agent = models.CharField(
        max_length=24,
        null=True,  # not answered call
    )

    staff_entry_binding = models.ForeignKey(
        CallCenterStaffEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    user_binding = models.ForeignKey(
        RequestUserPhoneBindingEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    meta_info = JSONField(null=True)

    class Meta:
        db_table = 'audiotele_call_stats'

        indexes = [
            models.Index(
                fields=['time_enter'],
                name='cc_audiotele_time_enter_idx',
            ),
            models.Index(
                fields=['phone'],
                name='cc_audiotele_phone_idx',
            ),
            models.Index(
                fields=['related_call_id'],
                name='cc_audiotele_related_call_id_idx',
            ),
        ]

    def __str__(self):
        return ('<AudioteleIncomingCallEntry: time_enter={}, agent={}, phone={}>'
                .format(self.time_enter, self.agent, self.phone))


class AudioteleCallTrackEntry(models.Model):
    id = models.UUIDField(default=uuid.uuid4, primary_key=True)

    call_entry = models.OneToOneField(
        AudioteleIncomingCallEntry,
        on_delete=models.SET_NULL,
        related_name='track',
        null=True,
    )

    file_name = models.TextField()  # call id, enter date and time (UTC +3), phone number

    file_attrs = JSONField(null=True)

    class Meta:
        db_table = 'audiotele_call_tracks'

        indexes = [
            models.Index(
                fields=['file_name'],
                name='cc_audiotele_call_track_file_name_idx',
            ),
        ]

    def _get_call_id_parts(self):
        plain_name = os.path.splitext(self.file_name)[0]
        call_id_parts = plain_name.split('_')
        return call_id_parts

    @property
    def call_id(self):
        call_id = self._get_call_id_parts()[0]
        return call_id

    @property
    def time_id(self):
        datetime_str_value = '_'.join(self._get_call_id_parts()[1:3])  # 2018-04-01_00-00-00
        return datetime_helper.localize(datetime.datetime.strptime(datetime_str_value, '%Y-%m-%d_%H-%M-%S'))

    @property
    def raw_phone(self):
        phone = self._get_call_id_parts()[3]
        return phone

    @property
    def phone(self):
        return phone_number_helper.normalize_phone_number(self.raw_phone)

    @property
    def mds_key(self):
        return os.path.join(str(self.id), self.file_name)
