import datetime
import enum
import os
import uuid

import pytz

from django.db import models
from django.contrib.postgres.fields import JSONField
from phonenumber_field.modelfields import PhoneNumberField

from cars.core.util import phone_number_helper
from cars.callcenter.models import CallCenterStaffEntry
from .request_user_phone_binding import RequestUserPhoneBindingEntry


class BeeperCallDirection(enum.Enum):
    INCOMING = 'i'
    OUTGOING = 'o'


class BeeperAgentEntry(models.Model):
    # agent id
    related_id = models.CharField(
        max_length=64,
    )

    # agent print name
    related_name = models.CharField(
        max_length=128,
        null=True,
    )

    staff_entry_binding = models.ForeignKey(
        CallCenterStaffEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    class Meta:
        db_table = 'beeper_operators'


class BeeperCallEntry(models.Model):
    direction = models.CharField(
        max_length=1,
        choices=[(x.value, x.name) for x in BeeperCallDirection],
    )

    time_enter = models.DateTimeField()

    time_connect = models.DateTimeField(null=True)  # not answered call

    time_exit = models.DateTimeField()

    is_answered = models.BooleanField()

    duration = models.PositiveIntegerField(null=True)  # not answered call

    phone = PhoneNumberField(null=True)

    internal_call_id = models.CharField(  # deprecated (previously: call data hash to fast comparison)
        max_length=64,
        null=True,
    )

    external_call_id = models.CharField(  # call_router_day + call_router_key or absent
        max_length=64,
        null=True,
    )

    beeper_agent_entry_binding = models.ForeignKey(
        BeeperAgentEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    user_binding = models.ForeignKey(
        RequestUserPhoneBindingEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    class Meta:
        db_table = 'beeper_call_stats'

        indexes = [
            models.Index(
                fields=['time_enter'],
                name='cc_beeper_time_enter_idx',
            ),
            models.Index(
                fields=['phone'],
                name='cc_beeper_phone_idx',
            ),
            models.Index(
                fields=['external_call_id'],
                name='cc_beeper_external_call_id_idx',
            ),
        ]

    def __str__(self):
        return '<BeeperCallEntry: time_enter={}, phone={}>'.format(self.time_enter, self.phone)


class BeeperCallTrackEntry(models.Model):
    id = models.UUIDField(default=uuid.uuid4, primary_key=True)

    call_entry = models.OneToOneField(
        BeeperCallEntry,
        on_delete=models.SET_NULL,
        related_name='track',
        null=True,
    )

    file_name = models.TextField()  # time_connect (UTC), call_router_day, call_router_key, [phone_number]

    file_attrs = JSONField(default=dict)

    class Meta:
        db_table = 'beeper_call_tracks'

    def _get_call_id_parts(self):
        plain_name = os.path.splitext(self.file_name)[0]
        call_id_parts = plain_name.split('_')
        return call_id_parts

    @property
    def time_id(self):
        # it's close to connect time or in range between time enter and time connect
        datetime_str_value = '_'.join(self._get_call_id_parts()[:2])  # 2018-04-01_00-00-00_3
        return pytz.UTC.localize(datetime.datetime.strptime(datetime_str_value, '%Y-%m-%d_%H-%M-%S'))

    @property
    def call_id(self):
        call_id_parts = self._get_call_id_parts()[3:5]
        call_id = '_'.join(call_id_parts) if all(p != '0' for p in call_id_parts) else None
        return call_id

    @property
    def raw_phone(self):
        call_id_parts = self._get_call_id_parts()
        phone = call_id_parts[5] if len(call_id_parts) == 6 else None
        return phone

    @property
    def phone(self):
        phone = '+7{}'.format(self.raw_phone) if self.raw_phone and len(self.raw_phone) == 10 else self.raw_phone
        return phone_number_helper.normalize_phone_number(phone)

    @property
    def mds_key(self):
        return os.path.join(str(self.id), self.file_name)
