import enum

from django.db import models
from django.contrib.postgres.fields import JSONField

from phonenumber_field.modelfields import PhoneNumberField


class SyncOrigin(enum.Enum):
    # internal callcenter
    CC_INTERNAL = 'i'

    CC_INTERNAL_STAT_MONITORING = 'cc_internal_stat_monitoring'
    CC_INTERNAL_OPERATOR_MONITORING = 'cc_internal_operator_monitoring'

    CC_INTERNAL_INCOMING_ENTRIES = 'cc_internal_incoming_entries'
    CC_INTERNAL_OUTGOING_ENTRIES = 'cc_internal_outgoing_entries'
    CC_INTERNAL_ALTAY_OUTGOING_ENTRIES = 'cc_internal_altay_outgoing_entries'

    # deprecated section, won't be used
    CC_INTERNAL_CARSHARING_UPDATE_TAGS = 's_i_carsharing_update_tags'
    CC_INTERNAL_CARSHARING_VIP_UPDATE_TAGS = 's_i_carsharing_vip_update_tags'
    CC_INTERNAL_OUTGOING_UPDATE_TAGS = 's_i_outgoing_update_tags'
    CC_INTERNAL_ALTAY_OUTGOING_UPDATE_TAGS = 'cc_internal_altay_outgoing_update_tags'
    # end of deprecated section

    # deprecated section, won't be used
    CC_INTERNAL_CARSHARING_TAGS_EXPORTING_DEPRECATED = 'cc_internal_carsharing_tag_exporting'
    CC_INTERNAL_CARSHARING_VIP_TAGS_EXPORTING_DEPRECATED = 'cc_internal_carsharing_vip_exporting'
    CC_INTERNAL_ALTAY_OUTGOING_TAGS_EXPORTING_DEPRECATED = 'cc_internal_altay_outgoing_tag_exporting'
    # end of deprecated section

    CC_INTERNAL_CARSHARING_TAGS_EXPORTING = 'cc_internal_carsharing_tags_exporting_ext'
    CC_INTERNAL_CARSHARING_VIP_TAGS_EXPORTING = 'cc_internal_carsharing_vip_tags_exporting_ext'
    CC_INTERNAL_ALTAY_OUTGOING_TAGS_EXPORTING = 'cc_internal_altay_outgoing_tags_exporting_ext'

    CC_INTERNAL_CARSHARING_TAGS_BINDING = 'cc_internal_carsharing_tags_binding'
    CC_INTERNAL_CARSHARING_VIP_TAGS_BINDING = 'cc_internal_carsharing_vip_tags_binding'
    CC_INTERNAL_OUTGOING_TAGS_BINDING = 'cc_internal_outgoing_tags_binding'
    CC_INTERNAL_ALTAY_OUTGOING_TAGS_BINDING = 'cc_internal_altay_outgoing_tags_binding'

    SERVICE_CC_INTERNAL_ASSIGNMENT_UPDATE = 's_cc_internal_assignment_update'

    CC_INTERNAL_USERS_BINDING = 'cc_internal_users_binding'

    # audiotele
    CC_AUDIOTELE = 'a'

    CC_AUDIOTELE_STAT_MONITORING = 'audiotele_operator_monitoring'

    CC_AUDIOTELE_TRACKS = 'audiotele_tracks'
    CC_AUDIOTELE_TRACK_BINDINGS = 'audiotele_track_bindings'

    CC_AUDIOTELE_INCOMING_TAGS_UPDATE = 's_audiotele_incoming_update_tags'
    CC_AUDIOTELE_OUTGOING_TAGS_UPDATE = 's_audiotele_outgoing_update_tags'

    # deprecated section, won't be used
    CC_AUDIOTELE_INCOMING_TAGS_EXPORTING_DEPRECATED = 'cc_audiotele_incoming_tags_exporting'
    CC_AUDIOTELE_OUTGOING_TAGS_EXPORTING_DEPRECATED = 'cc_audiotele_outgoing_tags_exporting'
    # end of deprecated section

    CC_AUDIOTELE_INCOMING_TAGS_EXPORTING = 'cc_audiotele_incoming_tags_exporting_ext'
    CC_AUDIOTELE_OUTGOING_TAGS_EXPORTING = 'cc_audiotele_outgoing_tags_exporting_ext'

    CC_AUDIOTELE_INCOMING_TAGS_BINDING = 'audiotele_incoming_tags_binding'
    CC_AUDIOTELE_OUTGOING_TAGS_BINDING = 'audiotele_outgoing_tags_binding'

    CC_AUDIOTELE_USERS_BINDING = 'cc_audiotele_users_binding'

    # tags
    # deprecated section, won't be used
    REQUEST_TAGS_EXPORTING_DEPRECATED = 'request_tags_exporting'
    # end of deprecated section

    REQUEST_TAGS_EXPORTING = 'request_tags_exporting_ext'
    REQUEST_TAGS_IMPORTING = 'request_tags_importing'

    # staff bindings
    SERVICE_CC_INTERNAL_IN_UPDATE_STAFF_BINDING = 's_i_in_update_staff_binding'
    SERVICE_CC_INTERNAL_OUT_UPDATE_STAFF_BINDING = 's_i_out_update_staff_binding'

    # chat2desk
    CHAT2DESK_USERS_BINDING = 'chat2desk_users_binding'
    CHAT2DESK_CHATS_EXPORTING = 'chat2desk_chats_exporting'

    # evacuations
    EVACUATION_USER_TAG_MONITORING = 'evacuation_user_tag_monitoring'
    EVACUATION_STS_TAG_MONITORING = 'evacuation_sts_tag_monitoring'
    EVACUATION_TAG_EVOLUTION_MONITORING = 'evacuation_tag_evolution_monitoring'

    # fines
    FINES_EXPORTING = 'fines_exporting'
    FINES_EXPORTING_EXT = 'fines_exporting_ext'
    YNDX_FINES = 'yndx_fines'
    FINES_CHARGING = 'fines_charging'
    FINES_PAYMENTS_CHECK = 'fine_payments_check'

    FINES_COLLECTING = 'fines_collecting'
    FINE_PHOTO_COLLECTING = 'fine_photo_collecting'

    # common
    CC_LOAD_BALANCE = 'cc_load_balance'

    # call data exporting
    CC_INTERNAL_INCOMING_DATA_EXPORTING = 'cc_internal_incoming_data_exporting'
    CC_INTERNAL_OUTGOING_DATA_EXPORTING = 'cc_internal_outgoing_data_exporting'
    CC_INTERNAL_ROUTING_DATA_EXPORTING = 'cc_internal_routing_data_exporting'
    CC_AUDIOTELE_DATA_EXPORTING = 'cc_audiotele_data_exporting'
    CC_AUDIOTELE_TRACK_DATA_EXPORTING = 'cc_audiotele_track_data_exporting'


class RoutingVerb(enum.Enum):
    INCOMING = 'INCOMING'
    DISTRIBUTE = 'DISTRIBUTE'
    XCONNECT = 'XCONNECT'


class RoutingTargetCallCenter(enum.Enum):
    AUDIOTELE = 'ATELE'
    BEEPER = 'BEEPER'
    NEXTCONTACT = 'NEXTCONTACT'
    YANDEX = 'YNDXCC'


class CallStatSyncStatus(models.Model):
    origin = models.CharField(max_length=64)

    # time of last data been synchronized or last sync time
    last_data_sync_time = models.DateTimeField()

    active_since = models.DateTimeField(null=True)
    active_until = models.DateTimeField(null=True)

    data = JSONField(null=True)

    class Meta:
        db_table = 'call_stat_sync_status'

    @property
    def extra_time_spans(self):
        return self.data.get('extra_time_spans', []) if self.data else []

    def append_extra_time_span(self, since, until):
        time_span = [since, until]

        assert all(isinstance(x, (int, float)) for x in time_span) and since <= until

        if self.data is None:
            self.data = {}

        extra_time_spans = self.data.get('extra_time_spans', [])
        extra_time_spans.append(time_span)
        self.data['extra_time_spans'] = extra_time_spans


class CallRoutingEntry(models.Model):
    time_id = models.DateTimeField()

    call_id = models.CharField(
        max_length=64,
    )

    verb = models.CharField(
        max_length=32,
    )

    source_cc = models.CharField(
        max_length=64,
        null=True,
    )

    # one of cars.core.telephony.TelephonyCallCenter values
    target_cc = models.CharField(
        max_length=64,
        null=True,
    )

    # could be FALLBACK_FROM_BEEPER, FALLBACK_FROM_ATELE - not used
    phone = PhoneNumberField(null=True)

    meta_info = JSONField(null=True)

    class Meta:
        db_table = 'call_routing_stats'

        indexes = [
            models.Index(
                fields=['time_id'],
                name='call_routing_time_id_idx',
            ),
            models.Index(
                fields=['phone'],
                name='call_routing_phone_idx',
            ),
            models.Index(
                fields=['call_id'],
                name='call_routing_call_id_idx',
            ),
        ]

    def __str__(self):
        return ('<CallRoutingEntry: target_cc={}, time_id={}, call_id={}>'
                .format(self.target_cc, self.time_id, self.call_id))
