import enum
import uuid

from django.db import models
from django.contrib.postgres.fields import JSONField
from phonenumber_field.modelfields import PhoneNumberField

from cars.users.models import User


class TagOrigin(enum.Enum):
    AUDIOTELE_INCOMING = 'ai'
    AUDIOTELE_OUTGOING = 'ao'
    CARSHARING = 'ii'  # internal incoming
    CARSHARING_VIP = 'iiv'  # internal incoming vip

    # CARSHARING_OUTGOING = 'io'  # deprecated
    CC_INTERNAL_ALTAY_OUTGOING = 'io'

    # real-time calls (not to markup)
    AUDIOTELE_REALTIME_INCOMING = 'audiotele_realtime_incoming'
    CC_INTERNAL_REALTIME_INCOMING = 'cc_internal_realtime_incoming'

    @property
    def is_call(self):
        return True

    @property
    def is_real_time(self):
        return self in (TagOrigin.AUDIOTELE_REALTIME_INCOMING, TagOrigin.CC_INTERNAL_REALTIME_INCOMING)

    @property
    def is_incoming_call(self):
        return self in (
            TagOrigin.CARSHARING, TagOrigin.CARSHARING_VIP, TagOrigin.AUDIOTELE_INCOMING,
            TagOrigin.CC_INTERNAL_REALTIME_INCOMING, TagOrigin.AUDIOTELE_REALTIME_INCOMING
        )

    @property
    def is_outgoing_call(self):
        return self in (TagOrigin.AUDIOTELE_OUTGOING, TagOrigin.CC_INTERNAL_ALTAY_OUTGOING)


class RequestOriginType(enum.Enum):
    AUDIOTELE_INCOMING = 'ai'
    AUDIOTELE_OUTGOING = 'ao'
    CARSHARING = 'ii'  # internal incoming
    CARSHARING_VIP = 'iiv'  # internal incoming vip

    # CARSHARING_OUTGOING = 'io'  # deprecated
    CC_INTERNAL_ALTAY_OUTGOING = 'io'

    CHAT2DESK_EXT_DIALOG_ID = 'chat2desk_ext_dialog_id'

    # real-time calls (not to markup)
    AUDIOTELE_REALTIME_INCOMING = 'audiotele_realtime_incoming'
    CC_INTERNAL_REALTIME_INCOMING = 'cc_internal_realtime_incoming'

    @property
    def is_call(self):
        return True

    @property
    def is_real_time(self):
        return self in (RequestOriginType.AUDIOTELE_REALTIME_INCOMING, RequestOriginType.CC_INTERNAL_REALTIME_INCOMING)

    @property
    def is_incoming_call(self):
        return self in (
            RequestOriginType.AUDIOTELE_INCOMING, RequestOriginType.CARSHARING, RequestOriginType.CARSHARING_VIP,
            RequestOriginType.CC_INTERNAL_REALTIME_INCOMING, RequestOriginType.AUDIOTELE_REALTIME_INCOMING
        )

    @property
    def is_outgoing_call(self):
        return self in (RequestOriginType.AUDIOTELE_OUTGOING, RequestOriginType.CC_INTERNAL_ALTAY_OUTGOING)


class RequestTagType(enum.Enum):
    OLD = 'old'
    NEW = 'new'


class CallTagCategory(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4)

    request = models.TextField()

    request_ru = models.TextField(null=True)  # temporary compatibility

    result = models.TextField()

    tag_origin = models.TextField(choices=[(x.value, x.name) for x in TagOrigin])

    class Meta:
        db_table = 'call_tag_category'


class RequestTagEntry(models.Model):
    COMMENT_MAX_PRINT_LENGTH = 128

    entry_id = models.UUIDField(primary_key=True, default=uuid.uuid4)

    submitted_at = models.DateTimeField()

    performer = models.ForeignKey(  # matched worker id; use generic relation to not restrict tag matching
        User,
        on_delete=models.SET_NULL,
        null=True
    )

    # tag description reference
    tag_id = models.UUIDField()  # (request, result) pair or new tag entry
    tag_type = models.TextField()  # is new type tag or old type one; can be RequestTagType but not restricted to

    # request relation reference
    request_id = models.TextField(null=True)  # internal or external call id
    request_origin = models.TextField()  # callcenter, queue, direction; can be RequestOriginType but not restricted to

    original_phone = PhoneNumberField(null=True)  # parsed phone; original phone is stored in meta info
    original_user_id = models.UUIDField(null=True)

    # used to specify request relation to another user directly (complaint)
    #   or indirectly (the user has multiple phones or share account)
    related_phone = PhoneNumberField(null=True)
    related_user_id = models.UUIDField(null=True)

    comment = models.TextField()  # comment

    meta_info = JSONField(null=True)  # original phone, OS, airport, model

    class Meta:
        db_table = 'request_tag'

        indexes = [
            models.Index(fields=['request_id', 'request_origin'], name='request_tag_request_id_origin_idx'),
        ]

    def __str__(self):
        return '<RequestTagEntry>'


class RequestTagEntryHistory(models.Model):
    history_event_id = models.AutoField(primary_key=True)

    history_user = models.ForeignKey(
        User,
        on_delete=models.SET_NULL,
        null=True,
        related_name='request_tag_history_history_user_id'
    )

    history_timestamp = models.IntegerField()

    history_action = models.TextField()

    entry_id = models.UUIDField()

    submitted_at = models.DateTimeField()

    performer = models.ForeignKey(
        User,
        on_delete=models.SET_NULL,
        null=True,
        related_name='request_tag_history_performer_id'
    )

    tag_id = models.UUIDField()
    tag_type = models.TextField()

    request_id = models.TextField(null=True)
    request_origin = models.TextField()

    original_phone = PhoneNumberField(null=True)
    original_user_id = models.UUIDField(null=True)

    related_phone = PhoneNumberField(null=True)
    related_user_id = models.UUIDField(null=True)

    comment = models.TextField()

    meta_info = JSONField(null=True)

    class Meta:
        db_table = 'request_tag_history'
