import enum

from django.db import models
from phonenumber_field.modelfields import PhoneNumberField

from cars.callcenter.models import CallCenterStaffEntry
from .request_user_phone_binding import RequestUserPhoneBindingEntry


class Chat2DeskOperatorRole(enum.Enum):
    ADMIN = 'admin'
    DELETED = 'deleted'
    DISABLED = 'disabled'
    OPERATOR = 'operator'
    SUPERVISOR = 'supervisor'
    UNCONFIRMED = 'unconfirmed'


class Chat2DeskTransportType(enum.Enum):
    TELEGRAM = 'telegram'
    WHATSAPP = 'whatsapp'


class Chat2DeskAttachmentType(enum.Enum):
    AUDIO = 'audio'
    PDF = 'pdf'
    PHOTO = 'photo'
    VIDEO = 'video'


class Chat2DeskEntryType(enum.Enum):
    FROM_CLIENT = 'from_client'
    TO_CLIENT = 'to_client'

    REQUEST_START = 'request_start'
    REQUEST_END = 'request_end'

    AUTOREPLY = 'autoreply'
    SYSTEM = 'system'


class Chat2DeskOperatorEntry(models.Model):
    related_id = models.IntegerField()  # chat2desk internal id

    related_staff_work_phone = models.IntegerField(null=True)

    staff_entry_binding = models.ForeignKey(
        CallCenterStaffEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    print_name = models.CharField(
        max_length=64,
        blank=True,
    )

    phone = PhoneNumberField(null=True)

    email = models.CharField(max_length=64)

    role = models.CharField(
        max_length=16,
        choices=[(x.value, x.name) for x in Chat2DeskOperatorRole],
    )

    class Meta:
        db_table = 'chat2desk_operators'

    def __str__(self):
        return ('<Chat2DeskOperatorEntry: staff_id={}, print_name={}>'
                .format(self.related_staff_work_phone, self.print_name))


class Chat2DeskClientEntry(models.Model):
    related_id = models.IntegerField()  # chat2desk internal id

    time_id = models.DateTimeField(null=True)  # user has been added, but it has no dialogs

    assigned_phone = PhoneNumberField(null=True)  # might also be blank

    # normalized first and last names; telegram identifier info
    extra_identification_info = models.CharField(
        max_length=255,
        blank=True,
        default='',
    )

    user_binding = models.ForeignKey(
        RequestUserPhoneBindingEntry,
        on_delete=models.SET_NULL,
        null=True
    )

    class Meta:
        db_table = 'chat2desk_clients'

        indexes = [
            models.Index(
                fields=['related_id'],
                name='chat2desk_clients_related_id_idx',
            ),
            models.Index(
                fields=['assigned_phone'],
                name='chat2desk_clients_assigned_phone_idx',
            ),
        ]

    def __str__(self):
        return '<Chat2DeskClientEntry: phone={}>'.format(self.assigned_phone)


class Chat2DeskDialogEntry(models.Model):
    related_client = models.ForeignKey(
        Chat2DeskClientEntry,
        on_delete=models.CASCADE,
    )

    related_id = models.IntegerField()  # chat2desk internal dialog id

    transport = models.CharField(
        max_length=8,
        choices=[(x.value, x.name) for x in Chat2DeskTransportType],
    )

    class Meta:
        db_table = 'chat2desk_dialogs'

        indexes = [
            models.Index(
                fields=['related_id'],
                name='chat2desk_dialogs_related_id_idx',
            ),
        ]

    def __str__(self):
        return ('<Chat2DeskDialogEntry: related_dialog_id={}, transport={}>'
                .format(self.related_id, self.transport))


class Chat2DeskMessageEntry(models.Model):
    MAX_URL_LENGTH = 512
    MAX_TEXT_LENGTH = 2048
    MAX_TEXT_PRINT_LENGTH = 30

    related_id = models.IntegerField()  # chat2desk internal id

    # note: every message is related to a dialog, however information about dialog cannot be obtained real-time
    #   moreover, every user has a unique dialog-transport pair, so related dialog can be identified surely

    related_dialog_id = models.IntegerField(null=True)  # chat2desk internal dialog id

    related_request_id = models.IntegerField(null=True)  # chat2desk internal request id

    time_id = models.DateTimeField()

    related_client = models.ForeignKey(
        Chat2DeskClientEntry,
        on_delete=models.CASCADE,
        null=True,
    )

    related_operator = models.ForeignKey(
        Chat2DeskOperatorEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    text = models.CharField(max_length=MAX_TEXT_LENGTH, blank=True)

    attachment_url = models.URLField(max_length=MAX_URL_LENGTH, null=True)

    attachment_type = models.CharField(
        max_length=8,
        choices=[(x.value, x.name) for x in Chat2DeskAttachmentType],
        null=True,
    )

    transport = models.CharField(
        max_length=8,
        choices=[(x.value, x.name) for x in Chat2DeskTransportType],
    )

    entry_type = models.CharField(
        max_length=16,
        choices=[(x.value, x.name) for x in Chat2DeskEntryType],
    )

    class Meta:
        db_table = 'chat2desk_messages'

        indexes = [
            models.Index(
                fields=['related_id'],
                name='chat2desk_messages_related_id_idx',
            ),
            models.Index(
                fields=['time_id'],
                name='chat2desk_messages_time_id_idx',
            ),
        ]

    def __str__(self):
        return ('<Chat2DeskMessageEntry: time_id={}, entry_type={}, message={}>'
                .format(self.time_id, self.entry_type, self.text[:self.MAX_TEXT_PRINT_LENGTH]))
