import enum

from django.db import models
from django.contrib.postgres.fields import JSONField
from phonenumber_field.modelfields import PhoneNumberField

from cars.callcenter.models import CallCenterStaffEntry
from .request_user_phone_binding import RequestUserPhoneBindingEntry


class InternalCCVerb(enum.Enum):
    ADD_MEMBER = 'ADDMEMBER'
    REMOVE_MEMBER = 'REMOVEMEMBER'

    ENTER_QUEUE = 'ENTERQUEUE'

    ABANDON = 'ABANDON'
    EXIT_EMPTY = 'EXITEMPTY'
    EXIT_WITH_TIMEOUT = 'EXITWITHTIMEOUT'

    CONNECT = 'CONNECT'
    RING_NO_ANSWER = 'RINGNOANSWER'

    COMPLETE_AGENT = 'COMPLETEAGENT'
    COMPLETE_CALLER = 'COMPLETECALLER'

    ATTENDED_TRANSFER = 'ATTENDEDTRANSFER'
    BLIND_TRANSFER = 'BLINDTRANSFER'

    _STUB = '_STUB'  # a sentinel value

    @classmethod
    def try_make(cls, value, default=None):
        try:
            return cls(value)
        except (TypeError, ValueError):
            return default if isinstance(default, InternalCCVerb) else InternalCCVerb._STUB

    @property
    def precedence(self):
        return InternalCCVerbPrecedence.get(self)

    def has_phone(self):
        return self is InternalCCVerb.ENTER_QUEUE

    def has_agent(self):
        return self.is_operator_action_entry() or self.is_decision_entry() or self.is_completion_entry()

    def is_operator_action_entry(self):
        return self in (InternalCCVerb.ADD_MEMBER, InternalCCVerb.REMOVE_MEMBER)

    def is_enter_entry(self):
        return self is InternalCCVerb.ENTER_QUEUE

    def is_decision_entry(self):
        return self in (InternalCCVerb.CONNECT, InternalCCVerb.RING_NO_ANSWER)

    def is_connect_entry(self):
        return self is InternalCCVerb.CONNECT

    def is_abandon_entry(self):
        return self in (InternalCCVerb.ABANDON, InternalCCVerb.EXIT_EMPTY, InternalCCVerb.EXIT_WITH_TIMEOUT)

    def is_transfer_entry(self):
        return self in (InternalCCVerb.ATTENDED_TRANSFER, InternalCCVerb.BLIND_TRANSFER)

    def is_completion_entry(self):
        return self in (InternalCCVerb.COMPLETE_AGENT, InternalCCVerb.COMPLETE_CALLER)


class InternalCCVerbPrecedence(object):
    default = -1

    # enter -> decision [-> decision...] -> completion / abandon
    precedence = {
        InternalCCVerb.ENTER_QUEUE: 0,
        InternalCCVerb.RING_NO_ANSWER: 10,
        InternalCCVerb.CONNECT: 11,
        InternalCCVerb.COMPLETE_AGENT: 20,
        InternalCCVerb.COMPLETE_CALLER: 20,
        InternalCCVerb.ABANDON: 20,
        InternalCCVerb.EXIT_EMPTY: 20,
        InternalCCVerb.EXIT_WITH_TIMEOUT: 20,
        InternalCCVerb.ATTENDED_TRANSFER: 20,
        InternalCCVerb.BLIND_TRANSFER: 20,
    }

    @classmethod
    def get(cls, value):
        return cls.precedence.get(value, cls.default)


class CallCenterEntry(models.Model):
    time_id = models.DateTimeField()

    # null or blank for ENTERQUEUE if 'anonymous' and for any entry except ENTERQUEUE
    phone = PhoneNumberField(null=True)

    # refer cars.core.telephony.TelephonyQueue
    queue_name = models.CharField(
        max_length=32,
    )

    verb = models.CharField(
        max_length=32,
    )

    call_id = models.CharField(
        max_length=64,
    )

    agent = models.CharField(
        max_length=24,
        null=True,
    )

    staff_entry_binding = models.ForeignKey(
        CallCenterStaffEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    user_binding = models.ForeignKey(
        RequestUserPhoneBindingEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    meta_info = JSONField(null=True)  # e.g. original phone

    class Meta:
        db_table = 'call_center_stats'

        indexes = [
            models.Index(
                fields=['time_id'],
                name='cc_internal_in_time_id_idx',
            ),
            models.Index(
                fields=['phone'],
                name='cc_internal_in_phone_idx',
            ),
            models.Index(
                fields=['call_id'],
                name='cc_internal_in_call_id_idx',
            ),
        ]

    @property
    def verb_instance(self):
        if getattr(self, '_verb_instance', None) is None:
            self._verb_instance = InternalCCVerb.try_make(self.verb)
        return self._verb_instance

    def __str__(self):
        return '<CallCenterEntry: verb={}, time_id={}, call_id={}>'.format(self.verb, self.time_id, self.call_id)


class OutgoingCallCenterEntry(models.Model):
    time_enter = models.DateTimeField()

    time_connect = models.DateTimeField(null=True)

    time_exit = models.DateTimeField()

    duration = models.PositiveIntegerField()

    phone = PhoneNumberField(null=True)

    # agent work phone
    agent = models.CharField(
        max_length=24,
    )

    staff_entry_binding = models.ForeignKey(
        CallCenterStaffEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    user_binding = models.ForeignKey(
        RequestUserPhoneBindingEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    class Meta:
        db_table = 'outgoing_call_center_stats'

        indexes = [
            models.Index(
                fields=['time_enter'],
                name='cc_internal_out_time_enter_idx',
            ),
            models.Index(
                fields=['phone'],
                name='cc_internal_out_phone_idx',
            ),
        ]

    def __str__(self):
        return ('<OutgoingCallCenterEntry: time_enter={}, agent={}, phone={}>'
                .format(self.time_enter, self.agent, self.phone))


class InternalCCAltayStatus(enum.Enum):
    SUCCESS = 'success'
    LINE_BUSY = 'line_is_busy'
    NO_RESPONSE = 'no_response'
    NOT_AVAILABLE = 'not_available'
    ANSWER_PHONE = 'answerphone'
    APARTMENT = 'apartment'


class InternalCCAltayNOCStatus(enum.Enum):
    ANSWER = 'answer'
    BUSY = 'busy'
    NO_ANSWER = 'noanswer'
    CHANNEL_UNAVAILABLE = 'chanunavail'
    CANCEL = 'cancel'
    CONGESTION = 'congestion'


class InternalCCAltayCallType(enum.Enum):
    formal_from_yandex = 'f'
    informal_from_customer = 'i'


# Altay entries
class CallCenterAltayOutgoingEntry(models.Model):
    external_id = models.BigIntegerField()

    status = models.TextField()  # InternalCCAltayStatus

    noc_status = models.TextField(null=True)  # InternalCCAltayNOCStatus

    time_enter = models.DateTimeField()

    time_connect = models.DateTimeField(null=True)

    time_exit = models.DateTimeField()

    duration = models.PositiveIntegerField(null=True)

    wait_duration = models.PositiveIntegerField(null=True)

    call_type = models.CharField(max_length=1, choices=[(x.value, x.name) for x in InternalCCAltayCallType], null=True)

    comment = models.TextField(default='')

    phone = PhoneNumberField(null=True)

    staff_entry_binding = models.ForeignKey(
        CallCenterStaffEntry,
        on_delete=models.SET_NULL,
        null=True,
    )

    has_record = models.BooleanField()

    meta_info = JSONField(null=True)  # original phone, other call type, user

    class Meta:
        db_table = 'call_center_altay_outgoing_stats'

        indexes = [
            models.Index(
                fields=['external_id'],
                name='cc_internal_altay_out_external_id_idx',
            ),
            models.Index(
                fields=['time_enter'],
                name='cc_internal_altay_out_time_enter_idx',
            ),
            models.Index(
                fields=['phone'],
                name='cc_internal_altay_out_phone_idx',
            ),
        ]

    def __str__(self):
        return (
            '<CallCenterAltayOutgoingEntry: '
            'external_id={}, status={}, noc_status={}, '
            'time_enter={}, time_connect={}, time_exit={}, '
            'duration={}, wait_duration={}, call_type={}, has_record={}>'
            .format(self.external_id, self.status, self.noc_status,
                    self.time_enter, self.time_connect, self.time_exit,
                    self.duration, self.wait_duration, self.call_type, self.has_record)
        )
