import enum
import uuid

from django.db import models
from phonenumber_field.modelfields import PhoneNumberField

from cars.users.models import User


class RequestUserPhoneBindingEntry(models.Model):

    class PhoneSourceType(enum.Enum):
        DIRECT_USER = 'du'  # from User entry
        USER_BINDING = 'ub'  # from user binding entry

        CHAT_BINDING = 'cb'  # from chat extra binding (may be overridden)
        CHAT_EXTRA_BINDING = 'ecb'  # from chat extra info

        CALL_TAG_BINDING = 'tb'  # from tag extra binding (may be overridden)
        CALL_TAG_EXTRA_BINDING = 'etb'  # from tag extra info

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)

    phone = PhoneNumberField()

    user = models.ForeignKey(
        User,
        on_delete=models.SET_NULL,
        null=True,
    )

    phone_source = models.CharField(
        max_length=4,
        choices=[(x.value, x.name) for x in PhoneSourceType],
        null=True,
    )

    class Meta:
        db_table = 'request_user_phone_binding'

    @property
    def is_verified(self):
        return self.phone_source in (self.PhoneSourceType.DIRECT_USER.value,
                                     self.PhoneSourceType.USER_BINDING.value, )

    @property
    def can_be_rebound(self):
        return self.phone_source in (None,
                                     self.PhoneSourceType.CHAT_BINDING.value,
                                     self.PhoneSourceType.CALL_TAG_BINDING.value, )

    def __str__(self):
        return '<RequestUserPhoneBindingEntry: phone={}, user={}>'.format(self.phone, self.user)
