import base64
import collections
import concurrent.futures
import itertools
import logging
import os
import sys
import time
import urllib3
import uuid

import requests


def setup():
    sys.path.append('../')
    sys.path.append('../../')
    sys.path.append('../../../')

    import django
    os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'cars.django.settings')
    django.setup()


if __name__ == '__main__':
    setup()

from django.db import transaction

from cars.core.history import HistoryManager
from cars.users.models import User, UserDataHistory, UserTag, UserTagHistory


LOGGER = logging.getLogger(__name__)


def make_requests_session(retries=3, backoff_factor=1, pool_connections=10, pool_maxsize=10):
    session = requests.Session()
    retry = urllib3.Retry(
        total=retries,
        backoff_factor=backoff_factor,
        status_forcelist=[429, 502, 503, 504],
    )
    adapter = requests.adapters.HTTPAdapter(
        max_retries=retry,
        pool_connections=pool_connections,
        pool_maxsize=pool_maxsize,
    )
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    session.headers.update({
        'Authorization': 'OAuth {}'.format(os.environ['CARSHARING_ROBOT_CARSHARING_TOKEN']),
    })
    return session


class UserHistoryManager(HistoryManager):
    def __init__(self):
        super().__init__(
            UserDataHistory,
            excluded_fields=('password', ),
            coercing_rules={'tags': (lambda x: (str(x)[:128] if x is not None else None))}
        )


class UserTagsMovingHelper(object):
    TagTransform = collections.namedtuple('TagTransform', ['destination_tag', 'remove_old_tag'])

    BASE_URL = (
        "https://prestable.carsharing.yandex.net/api/staff/user_tags/add?service=drive-frontend&version=v5.0.0"
    )

    LIST_USER_TAG_URL = (
        "https://prestable.carsharing.yandex.net/api/staff/user_tags/list?object_id={}service=drive-frontend&version=v5.0.0"
    )

    WORK_THREAD_COUNT = 8

    def __init__(self):
        self._tag_history_manager = HistoryManager(UserTagHistory)
        self._user_history_manager = UserHistoryManager()
        self._performer_id = self._init_performer_id()
        self._tag_transform = self._init_tag_transform()
        self._filtered_user_ids = self._init_filtered_user_ids()
        self._session = make_requests_session()

    def _init_performer_id(self):
        # default_performer = User.objects.get(id='ec65f4f0-8fa8-4887-bfdc-ca01c9906696')  # username='robot-carsharing'
        default_performer = User.objects.get(id='5dd9913e-dde1-4dfb-a179-408f6bbab86a')  # username='dkositsyn'
        assert default_performer is not None
        return str(default_performer.id)

    def _init_tag_transform(self):
        tag_transform = {
            'chargeback': self.TagTransform('user_character_chargeback', True),
            'critic': self.TagTransform('user_character_critic', True),
            'popular_person': self.TagTransform('user_character_popular', True),

            'violation_hard_1': self.TagTransform(None, True),
            'violation_hard_2': self.TagTransform(None, True),
            'violation_light_1': self.TagTransform(None, True),
            'violation_light_2': self.TagTransform(None, True),
            'violation_light_3': self.TagTransform(None, True),

            'might_be_bonus_scam': self.TagTransform('user_character_mb_bonus_scam', False),
            'might_be_fuel_scam': self.TagTransform('user_character_mb_fuel_scam', False),
            'violation_hard_3': self.TagTransform('blocked_by_support', False),
            'drunk_driver': self.TagTransform('user_character_drunk_driver', False),
            'drug_man': self.TagTransform('user_character_drug_man', False),
            'accident_run': self.TagTransform('blocked_accident_run', False),
            'blocked_chargeback': self.TagTransform('blocked_chargeback', False),

            # added 2019-03-20
            'account_transfer': self.TagTransform('blocked_account_transfer', False),
            'asked to delete': self.TagTransform('asked_to_delete', False),
            'asked_to_delete': self.TagTransform('asked_to_delete', False),
            'smoker': self.TagTransform('violation_donotknow_smoked', False),

            # added 2019-03-29
            'silver_client': self.TagTransform('user_character_silver', False),
            'gold_client': self.TagTransform('user_character_gold', False),
            'platinum_client': self.TagTransform('user_character_platinum', False),
        }

        note_tags = [
            'girl_with_fish',
            'https://st.yandex-team.ru/drivesup-151#1539766114000',
            'https://web.chat2desk.com/chat/my?dialogid=4045460',
            'https://xn--90adear.xn--p1ai/check/driver#7836497131+24.02.2018 прошла проверка',
            'https://гибдд.рф/check/driver права активны 19.10',
            'ву с экрана',
            'дубль - другой аккаунт, этот разблокировала',
            'замена паспорта',
            'замечен курящим в салоне ( есть фото)',
            'из представительства хёнде',
            'инвалид, голос искажен',
            'иностранное ву',
            'критик',
            'курит в авто',
            'меньше 21 года',
            'не завершать длительную бронь',
            'неправильная парковка. в зоне действия знака 3.27, еще и на газоне',
            'по https://гибдд.рф/check/driver права норм 19.10',
            'предоставлял фэйковые права приднестровья',
            'прислал страницу с фото продления внж https://web.chat2desk.com/chat/all?dialogid=3798285',
            'прошу позвонить. если еще раз оставит - блок',
            'стаж меньше 2 лет',
            'стаж с 1986',
            'стаж с 1991 года',
            'стаж с 1995',
            'стаж с 2006',
            'стаж с 2007',
            'стаж с 2008',
            'стаж с 2011',
            'стаж с 2013',
            'тверь, украл стс',
            'часть документов фото с экрана',
        ]

        for note_tag in note_tags:
            tag_transform[note_tag] = self.TagTransform('note', True)

        return tag_transform

    def _init_filtered_user_ids(self):
        return []

    def migrate_tags(self):
        for user in self._iter_user_data_to_transform():
            self._transform_selected_user_tags(user)

    def migrate_tags_async(self):
        with concurrent.futures.ThreadPoolExecutor(max_workers=self.WORK_THREAD_COUNT) as executor:
            futures = [
                executor.submit(self._transform_selected_user_tags, user)
                for user in self._iter_user_data_to_transform()
            ]

            for idx, future in zip(itertools.count(), concurrent.futures.as_completed(futures)):
                try:
                    future.result()
                except Exception as exc:
                    LOGGER.exception('future raised an exception: {}'.format(exc))

    def _iter_user_data_to_transform(self):
        for idx, user in enumerate(self._iter_users()):
            user_tags = user.tags or []
            has_tags_to_transform = any(t in self._tag_transform for t in user_tags)

            if has_tags_to_transform:
                yield user

            if not (idx + 1) % 10000:
                LOGGER.info('total processed users: {}'.format(idx + 1))

    def _iter_users(self):
        qs = User.objects.only('id', 'tags').order_by('id')

        if self._filtered_user_ids:
            qs = qs.filter(id__in=self._filtered_user_ids)

        return qs

    def _transform_selected_user_tags(self, user):
        try:
            tags_to_transform = list(t for t in user.tags if t in self._tag_transform)
            LOGGER.info('transforming tags {} for user {}'.format(tags_to_transform, user.id))

            with transaction.atomic(savepoint=False):
                for tag in tags_to_transform:
                    self._transform_selected_tag(user, tag)

                    user_to_update = User.objects.select_for_update().get(id=user.id)
                    self._update_user_tags(user_to_update)

        except Exception:
            LOGGER.exception('error processing user {}'.format(user.id))
            raise

    def _transform_selected_tag(self, user, tag):
        tag_transform = self._tag_transform[tag]
        destination_tag = tag_transform.destination_tag

        if destination_tag is not None:
            if True or not self._check_user_has_tag(user, destination_tag):  # toggle to False to check tag presence
                comment = tag if destination_tag == 'note' else ''
                self._make_tag(user, destination_tag, comment)
            else:
                LOGGER.info('user {} has tag {} already'.format(user.id, destination_tag))

    def _check_user_has_tag(self, user, destination_tag):
        tags = [x['tag'] for x in self._list_user_tags(user)]
        has_tag = (destination_tag in tags)
        return has_tag

    def _list_user_tags(self, user):
        r = self._session.get(self.LIST_USER_TAG_URL.format(user.id), verify=False, timeout=3)

        if r.status_code != 200:
            LOGGER.error('response for user {} has status {}: {}'.format(str(user.id), r.status_code, r.content))

        r.raise_for_status()

        return r.json()['records']

    def _make_tag(self, user, tag, comment):
        return self._make_tag_proxy(user, tag, comment)

    def _make_tag_proxy(self, user, tag, comment):
        raw_data = {'tag': tag, 'comment': comment, 'object_id': str(user.id)}
        r = self._session.post(self.BASE_URL, json=raw_data, verify=False, timeout=3)

        if r.status_code != 200:
            LOGGER.error('response for user {} has status {}: {}'.format(str(user.id), r.status_code, r.content))

        r.raise_for_status()

    def _make_tag_db(self, user, tag, comment):
        user_tag_instance = UserTag(
            tag_id=uuid.uuid4(),
            object=user,
            tag=tag,
            data=self._make_data(comment),
        )

        # atomicity is provided in the outer function
        user_tag_instance.save()
        self._tag_history_manager.add_entry(user_tag_instance, self._performer_id)

    def _make_data(self, text):
        out = b'\x04\x00\x00\x00\x08\x01\x10\x01'  # header

        if text:
            text_data = text.encode('utf-8')

            data_length = chr(len(text_data)).encode('utf-8')

            if len(text_data) >= 128:
                data_length_bytes_count = chr(len(text_data) // 128).encode('utf-8')
                data_length += data_length_bytes_count

            data = b'\x0a' + data_length + text_data
            out += data

        transformed_out = base64.b64encode(out).decode('utf-8')
        return transformed_out

    def _update_user_tags(self, user):
        # atomicity is provided in the outer function
        user.tags = [
            t for t in user.tags
            if t not in self._tag_transform or not self._tag_transform[t].remove_old_tag
        ]
        user.save()

        self._user_history_manager.update_entry(user, self._performer_id)


class UserCppTagsMovingHelper(object):
    TagTransform = collections.namedtuple('TagTransform', ['destination_tag', 'remove_old_tag'])

    ADD_USER_TAG_URL = (
        "https://prestable.carsharing.yandex.net/api/staff/user_tags/add"
    )

    LIST_USER_TAG_URL = (
        "https://prestable.carsharing.yandex.net/api/staff/user_tags/list?object_id={}"
    )

    WORK_THREAD_COUNT = 8

    def __init__(self):
        self._performer_id = self._init_performer_id()
        self._tag_transform = self._init_tag_transform()
        self._filtered_user_ids = self._init_filtered_user_ids()
        self._session = make_requests_session()

    def _init_performer_id(self):
        default_performer = User.objects.get(
            id='5dd9913e-dde1-4dfb-a179-408f6bbab86a'  # username='dkositsyn'; production
            # id='8098f2fa-3cf9-41c7-8117-a3e4db1d12a0'  # username='dkositsyn'; testing
        )
        assert default_performer is not None
        return str(default_performer.id)

    def _init_tag_transform(self):
        return {
            'asked_to_delete': self.TagTransform('blocked_to_delete', False),
        }

    def _init_filtered_user_ids(self):
        user_ids = UserTag.objects.filter(tag__in=self._tag_transform).values_list('object_id', flat=True)
        return user_ids

    def migrate_tags_async(self):
        with concurrent.futures.ThreadPoolExecutor(max_workers=self.WORK_THREAD_COUNT) as executor:
            futures = [
                executor.submit(self._transform_selected_user_tags, user)
                for user in self._iter_user_data_to_transform()
            ]

            for idx, future in zip(itertools.count(), concurrent.futures.as_completed(futures)):
                try:
                    future.result()
                except Exception as exc:
                    LOGGER.exception('future raised an exception: {}'.format(exc))

    def _iter_user_data_to_transform(self):
        yield from self._iter_users()

    def _iter_users(self):
        qs = User.objects.only('id').order_by('id')

        if self._filtered_user_ids:
            qs = qs.filter(id__in=self._filtered_user_ids)

        return qs

    def _transform_selected_user_tags(self, user):
        try:
            LOGGER.info('transforming for user {}'.format(user.id))

            user_tags = self._list_user_tags(user)

            tags_to_transform = (tag for tag in user_tags if tag['tag'] in self._tag_transform)

            for tag in tags_to_transform:
                self._transform_selected_tag(user, tag, user_tags)

        except Exception:
            LOGGER.exception('error processing user {}'.format(user.id))
            raise

    def _transform_selected_tag(self, user_instance, tag, user_tags):
        tag_transform = self._tag_transform[tag['tag']]
        destination_tag_name = tag_transform.destination_tag

        if destination_tag_name is not None:
            if not self._check_user_has_tag(destination_tag_name, user_tags):
                comment = tag.get('comment', '')
                self._make_tag(user_instance, destination_tag_name, comment)
            else:
                LOGGER.info('user {} has tag {} already'.format(user_instance.id, destination_tag_name))

        if tag_transform.remove_old_tag:
            raise NotImplementedError('cannot remove tags')

    def _list_user_tags(self, user):
        r = self._session.get(self.LIST_USER_TAG_URL.format(user.id), verify=False, timeout=3)

        if r.status_code != 200:
            LOGGER.error('response for user {} has status {}: {}'.format(str(user.id), r.status_code, r.content))

        r.raise_for_status()

        return r.json()['records']

    def _check_user_has_tag(self, tag_name, user_tags):
        return any(tag_name == tag['tag'] for tag in user_tags)

    def _make_tag(self, user, tag, comment):
        raw_data = {'tag': tag, 'comment': comment, 'object_id': str(user.id)}
        r = self._session.post(self.ADD_USER_TAG_URL, json=raw_data, verify=False, timeout=3)

        if r.status_code != 200:
            LOGGER.error('response for user {} has status {}: {}'.format(str(user.id), r.status_code, r.content))

        r.raise_for_status()


if __name__ == '__main__':
    LOGGER.addHandler(logging.StreamHandler())

    file_name = './move_user_tags_{}.log'.format(int(time.time()))
    LOGGER.addHandler(logging.FileHandler(file_name))

    helper = UserCppTagsMovingHelper()
    helper.migrate_tags_async()
