import logging

from rest_framework.fields import empty, IntegerField, CharField, SerializerMethodField
from rest_framework.exceptions import ValidationError
from rest_framework.serializers import as_serializer_error, ModelSerializer

from cars.django.serializers import BaseSerializer, TimestampField
from cars.callcenter.core import StaffInfoHelper
from cars.callcenter.serializers import RequestUserSerializer
from cars.request_aggregator.models import Chat2DeskDialogEntry, Chat2DeskMessageEntry
from cars.users.models import User


LOGGER = logging.getLogger(__name__)


class Chat2DeskMessagesListArgumentsSerializer(BaseSerializer):
    staff_info_helper = StaffInfoHelper.make_default()

    dialog_id = IntegerField(
        required=True,
    )

    since = TimestampField(
        required=False,
    )

    until = TimestampField(
        required=False,
    )

    def run_validation(self, data=empty):
        try:
            value = super().run_validation(data)
            self._coerce_dialog(value)
        except ValidationError as original_exc:
            raise ValidationError(detail=as_serializer_error(original_exc))
        except (TypeError, ValueError) as original_exc:
            raise ValidationError(detail=as_serializer_error(ValidationError())) from original_exc

        return value

    def _coerce_dialog(self, data):
        dialog_id = data['dialog_id']

        dialog_entry = (
            Chat2DeskDialogEntry.objects.select_related('related_client')
            .filter(related_id=dialog_id).first()
        )

        if dialog_entry is None:
            raise ValidationError(detail='no dialog with id "{}" found'.format(dialog_id))

        data['dialog_entry'] = dialog_entry

        client_entry = dialog_entry.related_client
        data['client_entry'] = client_entry

        if client_entry.assigned_phone is not None:
            user = (
                User.objects.filter(phone=client_entry.assigned_phone)
                .order_by('status').first()  # make "active" first if exists
            )
        else:
            user = None

        data['user_entry'] = user

        formatted_user_entry = RequestUserSerializer(user).data

        if formatted_user_entry['phone'] is None and client_entry.assigned_phone is not None:
            formatted_user_entry['phone'] = str(client_entry.assigned_phone)

        data['formatted_user_entry'] = formatted_user_entry

        data['transport'] = dialog_entry.transport


class Chat2DeskMessagesModelSerializer(ModelSerializer):
    staff_info_helper = StaffInfoHelper.make_default()

    id = IntegerField()
    related_id = IntegerField()

    dialog_id = SerializerMethodField()
    request_id = SerializerMethodField()

    entry_type = CharField()

    timestamp = SerializerMethodField()

    client = SerializerMethodField()
    operator = SerializerMethodField()

    content = SerializerMethodField()
    content_type = SerializerMethodField()

    transport = CharField()

    class Meta:
        model = Chat2DeskMessageEntry

        fields = [
            'id',
            'related_id',
            'dialog_id',
            'request_id',
            'entry_type',
            'timestamp',
            'client',
            'operator',
            'content',
            'content_type',
            'transport',
        ]

    def get_dialog_id(self, obj):
        dialog_id = self.context['request'].arguments['dialog_id']
        return dialog_id

    def get_request_id(self, obj):
        return obj.related_request_id

    def get_timestamp(self, obj):
        return TimestampField().to_representation(obj.time_id)

    def get_client(self, obj):
        formatted_user_entry = self.context['request'].arguments['formatted_user_entry']
        return formatted_user_entry

    def get_operator(self, obj):
        operator_entry = obj.related_operator
        staff_entry = operator_entry.staff_entry_binding if operator_entry is not None else None
        return self.staff_info_helper.format_staff_entry(staff_entry)

    def get_content(self, obj):
        return obj.attachment_url or obj.text

    def get_content_type(self, obj):
        return obj.attachment_type or 'text'
