import collections
import enum
import json
import logging

from rest_framework.fields import empty, IntegerField, CharField, ListField, ChoiceField, UUIDField
from rest_framework.exceptions import ValidationError
from rest_framework.serializers import as_serializer_error

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.core.util import datetime_helper
from cars.django.serializers import BaseSerializer, PhoneNumberField, TimestampField
from cars.settings import REQUEST_AGGREGATOR as settings
from cars.callcenter.core import StaffInfoHelper
from cars.users.models import User

from .common_validators import (
    OneOfValidator, AnyOfValidator, TimeRangeValidator, FieldTimeRangeValidator
)


LOGGER = logging.getLogger(__name__)


class EntriesListArgumentsSerializer(BaseSerializer):
    saas_client = SaasDriveAdminClient.from_settings()
    staff_info_helper = StaffInfoHelper.make_default()

    class Meta:
        validators = [
            OneOfValidator(
                AnyOfValidator('since', 'until', is_required=False),
                AnyOfValidator('time_start', 'time_end', is_required=False),
                is_required=False
            ),
            OneOfValidator('agent_username', 'agent_user_id', is_required=False),
            OneOfValidator('phone_number', 'uid', 'user_id', is_required=False),
        ]

        post_validators = [
            AnyOfValidator('time_range', 'phone_number', 'staff_entry_binding', 'limit'),
            FieldTimeRangeValidator(
                ('time_range', 'phone_number', 'staff_entry_binding', 'limit'),
                collections.OrderedDict([
                    (
                        ('limit', 'phone_number', ),  # only time range and staff binding present
                        TimeRangeValidator(settings['common']['max_staff_binding_request_time_range'])
                    ),
                    (
                        ('limit', 'phone_number', 'staff_entry_binding',),  # only time range presents
                        TimeRangeValidator(settings['common']['max_raw_request_time_range'])
                    ),
                    (
                        ('limit', 'phone_number', 'time_range', ),  # only staff binding presents
                        TimeRangeValidator(settings['common']['max_staff_binding_request_time_range'])
                    ),
                ])
            ),
        ]

    # user filter
    phone_number = PhoneNumberField(
        required=False,
    )

    uid = IntegerField(
        min_value=0,
        required=False,
    )

    user_id = UUIDField(
        required=False,
    )

    # time range filter
    since = TimestampField(
        required=False,
    )

    until = TimestampField(
        required=False,
    )

    time_start = TimestampField(
        required=False,
    )

    time_end = TimestampField(
        required=False,
    )

    # agent filter
    agent_username = CharField(
        min_length=2,
        max_length=64,
        required=False,
    )

    agent_user_id = UUIDField(
        required=False
    )

    # limit results

    limit = IntegerField(
        required=False,
        min_value=1,
        default=20  # any not None value to apply pagination thereafter
    )

    def run_validation(self, data=empty):
        try:
            value = super().run_validation(data)

            self._coerce_user(value)
            self._coerce_time_range_params(value)
            self._coerce_agent(value)

            post_validators = getattr(self.Meta, 'post_validators', [])
            for validator in post_validators[:]:
                validator(value)

        except ValidationError as original_exc:
            raise ValidationError(detail=as_serializer_error(original_exc))
        except (TypeError, ValueError) as original_exc:
            raise ValidationError(detail=as_serializer_error(ValidationError())) from original_exc

        LOGGER.info(
            'original filters: {}, transformed filters: {}'
            .format(json.dumps(data, default=str), json.dumps(value, default=str))
        )

        return value

    def _coerce_user(self, data):
        if 'uid' in data:
            user_filter = {'uid': data['uid']}
            require_user_presence = True
        elif 'user_id' in data:
            user_filter = {'id': str(data['user_id'])}
            require_user_presence = True
        elif 'phone_number' in data:
            # phone number may not present in the db
            user_filter = {'phone': data['phone_number']}
            require_user_presence = False
        else:
            return

        user = User.objects.filter(**user_filter).order_by('status').first()  # make "active" first if exists

        if user is not None:
            is_user_absent = self._check_user_deleted(user, self.context['request'])
        else:
            is_user_absent = True

        if is_user_absent:
            if require_user_presence:
                raise ValidationError(detail='no user with traits "{}" exists'.format(user_filter))
        else:
            data['user'] = user

            if user.phone is not None and 'phone_number' not in data:
                data['phone_number'] = str(user.phone)

    def _check_user_deleted(self, user, request):
        is_user_deleted = (user.status == 'deleted')
        is_user_deleted = is_user_deleted or not self.saas_client.check_access_to_deleting_user(user.id, request)
        return is_user_deleted

    def _coerce_time_range_params(self, data):
        if 'time_start' in data:
            data.setdefault('since', data.pop('time_start'))

        if 'time_end' in data:
            data.setdefault('until', data.pop('time_end'))

        if 'since' not in data:
            return

        data.setdefault('until', datetime_helper.utc_now())

        since, until = data['since'], data['until']

        data['time_range'] = (since, until)

    def _coerce_agent(self, data):
        if 'agent_user_id' in data:
            user_id = str(data['agent_user_id'])
            user = User.objects.filter(id=user_id).first()

            if user is None:
                raise ValidationError(detail='user with id "{}" does not exist'.format(user_id))

            agent_username = user.username
            agent_filter = {'user_id': user_id}

        elif 'agent_username' in data:
            agent_username = data['agent_username']
            agent_filter = {'username': agent_username}

        else:
            return

        value = self.staff_info_helper.get_agent_entry(**agent_filter)

        if value is not None:
            data['staff_entry_binding'] = value
        else:
            raise ValidationError(detail='user "{}" is not treated as operator'.format(agent_username))


class RequestType(enum.Enum):
    mail = 'mail'
    messenger = 'messenger'
    other = 'other'
    phone = 'phone'
    push = 'push'
    sms = 'sms'


class AggregatedEntriesListArgumentsSerializer(EntriesListArgumentsSerializer):
    request_type = ListField(
        child=ChoiceField(choices=[x.value for x in RequestType]),
        required=False,
    )


AggregatedResponseEntry = collections.namedtuple(
    'AggregatedResponseEntry',
    (
        'time_enter', 'time_exit', 'duration', 'duration_print',
        'connect_trial_count', 'time_connect', 'status',
        'request', 'operators', 'tags', 'phone', 'user',
        'message',  # deprecated
        'data', 'data_url',
     ),
)


class CallTrackArgumentsSerializer(BaseSerializer):
    track_id = CharField(required=True, max_length=36)  # either uuid or unsigned integer
