import collections
import datetime

from rest_framework.exceptions import ValidationError


class NumberOfValidator(object):
    def __init__(self, *fields, min_field_count=None, max_field_count=None, is_required=True):
        self._fields = fields

        self._min_field_count = min_field_count
        self._max_field_count = max_field_count

        self._is_required = is_required

    @property
    def fields(self):
        return [
            f if not isinstance(f, NumberOfValidator) else f.fields
            for f in self._fields
        ]

    def __call__(self, data, *, suppress_errors=False):
        number_of_fields = sum(
            1 for f in self._fields if (
                (data.get(f, None) is not None) if not isinstance(f, NumberOfValidator)
                else f(data, suppress_errors=True)
            )
        )

        if not self._is_required and not number_of_fields:
            return False  # group is absent

        if (
                (self._min_field_count is not None and number_of_fields < self._min_field_count) or
                (self._max_field_count is not None and number_of_fields > self._max_field_count)
        ):
            if not suppress_errors:
                if (
                        self._min_field_count is not None and
                        self._max_field_count is not None and
                        self._min_field_count == self._max_field_count
                ):
                    admissible_count = 'exactly {}'.format(self._max_field_count)
                else:
                    admissible_count = 'in range from {} to {}'.format(self._min_field_count, self._max_field_count)

                raise ValidationError(detail=(
                    'number of different fields "{}" provided can be {}'.format(self.fields, admissible_count)
                ))
            else:
                return False

        return True  # group presents


class OneOfValidator(NumberOfValidator):
    def __init__(self, *fields, is_required=True):
        super().__init__(*fields, min_field_count=1, max_field_count=1, is_required=is_required)


class AnyOfValidator(NumberOfValidator):
    def __init__(self, *fields, is_required=True):
        super().__init__(*fields, min_field_count=1, max_field_count=len(fields), is_required=is_required)


class AllOfValidator(NumberOfValidator):
    def __init__(self, *fields, is_required=True):
        super().__init__(*fields, min_field_count=len(fields), max_field_count=len(fields), is_required=is_required)


class TimeRangeValidator(object):
    def __init__(self, max_timedelta=None, time_range_name='time_range'):
        self._time_range_name = time_range_name

        assert max_timedelta is None or isinstance(max_timedelta, datetime.timedelta)
        self._max_timedelta = max_timedelta

    def __call__(self, data):
        if self._time_range_name not in data:
            raise ValidationError(detail=(
                'time range is required to be set in addition to request params'
            ))

        since, until = data[self._time_range_name]
        max_timedelta = self._max_timedelta

        if since > until:
            raise ValidationError(detail=(
                'time end "{}" must be greater or equal than time start "{}"'.format(since, until)
            ))

        if max_timedelta is not None and until > since + max_timedelta:
            actual_value, max_admissible_value = (until - since).total_seconds(), max_timedelta.total_seconds()
            raise ValidationError(detail=(
                'time end "{}" must not be greater than time start "{}" by more than {} seconds (actual: {})'
                .format(since, until, max_admissible_value, actual_value)
            ))


class FieldTimeRangeValidator(object):
    def __init__(self, required_fields, absent_field_validator_mapping):
        assert isinstance(absent_field_validator_mapping, collections.OrderedDict)
        assert all(isinstance(v, TimeRangeValidator) for v in absent_field_validator_mapping.values())

        self._required_fields = required_fields
        self._field_validator_mapping = absent_field_validator_mapping

    def __call__(self, data):
        for absent_fields, validator in self._field_validator_mapping.items():
            are_requested_fields_absent = all(
                data.get(f, None) is None
                for f in absent_fields
            )

            are_required_fields_present = all(
                data.get(f, None) is not None
                for f in self._required_fields
                if f not in absent_fields
            )

            if are_requested_fields_absent and are_required_fields_present:
                validator(data)
                break
