import logging

from django.utils import translation
from rest_framework.fields import empty, CharField, ChoiceField, UUIDField, SerializerMethodField
from rest_framework.exceptions import ValidationError
from rest_framework.serializers import ModelSerializer, as_serializer_error

import cars.settings

from cars.django.serializers import BaseSerializer, PhoneNumberField, TimestampField

from cars.core.telephony import TelephonyQueue
from cars.core.util import datetime_helper, phone_number_helper

from cars.callcenter.serializers import RequestUserSerializer

from cars.request_aggregator.models.internal_cc_stats import (
    CallCenterEntry, CallCenterAltayOutgoingEntry, InternalCCVerb
)
from cars.request_aggregator.models.call_center_common import CallRoutingEntry, RoutingVerb
from cars.request_aggregator.models.audiotele_stats import AudioteleIncomingCallEntry, AudioteleCallDirection
from cars.request_aggregator.models.call_tags import CallTagCategory, RequestTagEntry, RequestTagType, RequestOriginType

from cars.users.models import User


LOGGER = logging.getLogger(__name__)


class RequestOriginField(ChoiceField):
    default_error_messages = {
        'invalid': translation.ugettext_lazy('"{value}" is not a valid request origin.'),
    }

    def __init__(self, **kwargs):
        choices = [x.name for x in RequestOriginType]
        super().__init__(choices, **kwargs)

    def to_internal_value(self, data):
        try:
            value = RequestOriginType[data]  # match by name
        except Exception:
            self.fail('invalid', value=data)
        else:
            return value

    def to_representation(self, value):
        try:
            origin_type = RequestOriginType(value)
            representation = origin_type.name
        except (TypeError, ValueError):
            self.fail('invalid', value=value)
        return representation


class RequestTagTypeField(ChoiceField):
    default_error_messages = {
        'invalid': translation.ugettext_lazy('"{value}" is not a valid request tag type.'),
    }

    def __init__(self, **kwargs):
        choices = [x.name for x in RequestTagType]
        super().__init__(choices, **kwargs)

    def to_internal_value(self, data):
        try:
            value = RequestTagType(data)  # match by value
        except Exception:
            self.fail('invalid', value=data)
        else:
            return value

    def to_representation(self, value):
        try:
            tag_type = RequestTagType[value]
            representation = tag_type.value
        except (TypeError, ValueError):
            self.fail('invalid', value=value)
        return representation


class RequestTagCategoryListArgumentsSerializer(BaseSerializer):
    request_origin = RequestOriginField(required=True)
    tag_type = RequestTagTypeField(default=RequestTagType.OLD)


class RequestTagCategorySerializer(BaseSerializer):
    id = UUIDField()
    type = SerializerMethodField()
    description = SerializerMethodField()
    order = SerializerMethodField()

    class Meta:
        model = CallTagCategory
        fields = [
            'id',
            'type',
            'description',
        ]

    def get_type(self, obj):
        return RequestTagType.OLD.value

    def get_description(self, obj):
        return self.context['tag_description_helper'].get_tag_category_description(obj, self.get_type(obj))

    def get_order(self, obj):
        return self.context['tag_description_helper'].get_tag_category_order(obj, self.get_type(obj))


class RequestTreeTagCategorySerializer(BaseSerializer):
    id = SerializerMethodField()
    type = SerializerMethodField()
    description = SerializerMethodField()
    order = SerializerMethodField()

    class Meta:
        fields = [
            'id',
            'type',
            'description',
        ]

    def get_id(self, obj):
        return obj['id']

    def get_type(self, obj):
        return RequestTagType.NEW.value

    def get_description(self, obj):
        return self.context['tag_description_helper'].get_tag_category_description(obj, self.get_type(obj))

    def get_order(self, obj):
        return self.context['tag_description_helper'].get_tag_category_order(obj, self.get_type(obj))


def _process_request(request_id, request_origin):
    assert isinstance(request_origin, RequestOriginType)
    filters = {'id': request_id}

    related_request_ids = None

    if request_origin in (RequestOriginType.CARSHARING, RequestOriginType.CARSHARING_VIP):
        if request_origin == RequestOriginType.CARSHARING_VIP:
            filters['queue_name'] = TelephonyQueue.CARSHARING_VIP.value
        request = CallCenterEntry.objects.get(**filters)

        related_request_ids = list(
            CallCenterEntry.objects.using(cars.settings.DB_RO_ID)
            .filter(call_id=request.call_id, verb=InternalCCVerb.ENTER_QUEUE.value)
            .values_list('id', flat=True)
        )

    elif request_origin == RequestOriginType.AUDIOTELE_INCOMING:
        filters['direction'] = AudioteleCallDirection.INCOMING.value
        request = AudioteleIncomingCallEntry.objects.get(**filters)

        related_request_ids = list(
            AudioteleIncomingCallEntry.objects.using(cars.settings.DB_RO_ID)
            .filter(related_call_id=request.related_call_id)
            .values_list('id', flat=True)
        )

    elif request_origin == RequestOriginType.AUDIOTELE_OUTGOING:
        filters['direction'] = AudioteleCallDirection.OUTGOING.value
        request = AudioteleIncomingCallEntry.objects.get(**filters)

    elif request_origin == RequestOriginType.CC_INTERNAL_ALTAY_OUTGOING:
        request = CallCenterAltayOutgoingEntry.objects.get(**filters)

    elif request_origin == RequestOriginType.CC_INTERNAL_REALTIME_INCOMING:
        request = CallCenterEntry.objects.filter(call_id=request_id, verb=InternalCCVerb.ENTER_QUEUE.value).first()

        if request is None:
            incoming_call_routing_entry = (
                CallRoutingEntry.objects
                .filter(call_id=request_id, verb=RoutingVerb.INCOMING.value)
                .first()
            )

            if incoming_call_routing_entry is not None:
                request_data = {
                    'queue_name': 'carsharing_dynamic',
                    'call_id': request_id,
                    'time_id': incoming_call_routing_entry.time_id,
                    'phone': incoming_call_routing_entry.phone,
                    'verb': InternalCCVerb.ENTER_QUEUE.value,
                    'meta_info': incoming_call_routing_entry.meta_info,
                }
                request = CallCenterEntry.objects.create(**request_data)
            else:
                request_data = {
                    'queue_name': 'carsharing_dynamic',
                    'call_id': request_id,
                    'time_id': datetime_helper.now(),
                    'phone': incoming_call_routing_entry.phone,
                    'verb': InternalCCVerb.ENTER_QUEUE.value,
                }
                request = CallCenterEntry.objects.create(**request_data)

        request_id = request.id

        if request.queue_name == TelephonyQueue.CARSHARING_VIP.value:
            request_origin = RequestOriginType.CARSHARING_VIP
        else:
            request_origin = RequestOriginType.CARSHARING

    elif request_origin == RequestOriginType.AUDIOTELE_REALTIME_INCOMING:
        request = AudioteleIncomingCallEntry.objects.filter(related_call_id=request_id).first()

        if request is None:
            raise ValueError('request is not found')

        request_id = request.id
        request_origin = RequestOriginType.AUDIOTELE_INCOMING

    else:
        raise ValueError('request is not found')

    return request, request_origin, request_id, related_request_ids


class RequestTagListArgumentsSerializer(BaseSerializer):
    request_origin = RequestOriginField(required=True)
    request_id = CharField(max_length=128, allow_blank=False, required=True)

    def run_validation(self, data=empty):
        try:
            value = super().run_validation(data)
            self._coerce_request(value)

        except ValidationError as original_exc:
            raise ValidationError(detail=as_serializer_error(original_exc))
        except (TypeError, ValueError) as original_exc:
            raise ValidationError(detail=as_serializer_error(ValidationError())) from original_exc

        return value

    def _coerce_request(self, data):
        request = related_request_ids = None

        request_id, request_origin = data['request_id'], data['request_origin']

        try:
            request, request_origin, request_id, related_request_ids = _process_request(request_id, request_origin)
        except Exception as exc:
            raise ValidationError(detail=(
                'no request with id "{}" and origin "{}" exists'
                .format(request_id, request_origin.name)
            ))

        # update values (they may be changed)
        data['request_id'], data['request_origin'] = request_id, request_origin
        data['request'] = request
        data['related_request_ids'] = related_request_ids


class RequestTagAssignmentArgumentsSerializer(BaseSerializer):
    # extra fields provided: tag, request, original_user, original_phone, related_user

    entry_id = UUIDField(required=False)

    tag_id = UUIDField(required=True)
    tag_type = ChoiceField(choices=[x.value for x in RequestTagType], default=RequestTagType.NEW.value)  # by value

    request_origin = RequestOriginField(required=True)
    request_id = CharField(max_length=128, allow_blank=False, required=False, default=None, allow_null=True)

    original_user_trait = CharField(max_length=256, required=False, default=None, allow_null=True)
    original_user_id = UUIDField(required=False, default=None, allow_null=True)

    related_phone = PhoneNumberField(required=False, default=None, allow_null=True)
    related_user_id = UUIDField(required=False, default=None, allow_null=True)

    comment = CharField(max_length=4096, allow_blank=True, required=False, default='')

    def run_validation(self, data=empty):
        try:
            value = super().run_validation(data)

            self._coerce_tag(value)
            self._coerce_original_user(value)
            self._coerce_related_user(value)
            self._coerce_request(value)

            if not (value['request'] or value['original_user_trait'] or value['original_user_id']):
                raise ValidationError(
                    detail='at least one of request id, original user trait or id has to be specified'
                )

        except ValidationError as original_exc:
            raise ValidationError(detail=as_serializer_error(original_exc))
        except (TypeError, ValueError) as original_exc:
            raise ValidationError(detail=as_serializer_error(ValidationError())) from original_exc

        return value

    def _coerce_tag(self, data):
        tag_id, tag_type = data['tag_id'], data['tag_type']

        try:
            tag_category_entry = self.context['tag_description_helper'].get_tag_category_entry(tag_id, tag_type)
            data['tag_category_entry'] = tag_category_entry
        except Exception as exc:
            raise ValidationError(detail='no tag category with id {} and type {} found'.format(tag_id, tag_type))

        tag_entry = None
        entry_id = data.get('entry_id', None)

        if entry_id is not None:
            tag_entry = RequestTagEntry.objects.filter(entry_id=entry_id).first()
            if tag_entry is None:
                raise ValidationError(detail='no tag entry with id {} found'.format(entry_id))

        data['tag_entry'] = tag_entry

    def _coerce_original_user(self, data):
        user_id_field_name = 'original_user_id'
        user_trait_field_name = 'original_user_trait'
        user_phone_field_name = 'original_phone'

        user = self._get_user(data, user_id_field_name, user_trait_field_name)

        request_origin = data['request_origin']
        assert isinstance(request_origin, RequestOriginType)

        if request_origin.is_call:
            user_trait = data.get(user_trait_field_name, None)
            user_phone = phone_number_helper.normalize_phone_number(user_trait)

            if user_trait and user_phone is None:
                raise ValidationError(detail=(
                    'value "{}" provided in field "{}" cannot be interpreted as a phone'
                    .format(user_trait, user_trait_field_name)
                ))
        else:
            user_phone = None

        user_field_name = user_id_field_name[:-len('_id')]
        data[user_field_name] = user
        data[user_phone_field_name] = user_phone

    def _coerce_related_user(self, data):
        user_id_field_name = 'related_user_id'
        user_trait_field_name = 'related_phone'

        user = self._get_user(data, user_id_field_name, user_trait_field_name)

        user_field_name = user_id_field_name[:-len('_id')]
        data[user_field_name] = user

    def _get_user(self, data, user_id_field_name, user_trait_field_name):
        user_id = data.get(user_id_field_name, None)

        if user_id is not None:
            try:
                user = User.objects.get(id=user_id)
            except Exception as exc:
                raise ValidationError(detail='no user with id "{}" exists'.format(user_id))

            if data.get(user_trait_field_name, None) is None:
                LOGGER.warning(
                    'user trait field "{}" is not set explicitly however user id {} does'
                    .format(user_trait_field_name, user_id_field_name)
                )
        else:
            user = None

        return user

    def _coerce_request(self, data):
        request = related_request_ids = None

        request_origin = data['request_origin']
        # assert isinstance(request_origin, RequestOriginType) - field guarantees that restriction

        if request_origin.is_real_time and data.get('request_id') is None:
            # artificial restriction, there is no problems to mark the request
            raise ValidationError(detail=(
                'request id is required for real-time requests (request origin - {})'.format(request_origin.name)
            ))

        if data.get('request_id') is not None:
            request_id = data['request_id']

            try:
                request, request_origin, request_id, related_request_ids = _process_request(request_id, request_origin)
            except Exception as exc:
                raise ValidationError(detail=(
                    'no request with id "{}" and origin "{}" exists'
                    .format(request_id, request_origin.name)
                ))

            if request_origin.is_call and request.phone is not None:
                request_phone = str(request.phone)
                original_phone = data.get('original_phone', None)

                if original_phone is None:
                    raise ValidationError(detail=(
                        'original phone is not provided however there is a request with phone "{}"'
                        .format(request_phone)
                    ))

                if request_phone != original_phone:
                    LOGGER.warning(
                        'original phone "{}" differs from request phone "{}" for request id {}'
                        .format(original_phone, request_phone, request_id)
                    )

            # update values (they may be changed)
            data['request_id'] = request_id
            data['request_origin'] = request_origin

        data['request'] = request
        data['related_request_ids'] = related_request_ids


class RequestTagRemovalArgumentsSerializer(BaseSerializer):
    entry_id = UUIDField(required=True)

    def run_validation(self, data=empty):
        try:
            value = super().run_validation(data)
            self._coerce_tag_entry(value)
        except ValidationError as original_exc:
            raise ValidationError(detail=as_serializer_error(original_exc))
        except (TypeError, ValueError) as original_exc:
            raise ValidationError(detail=as_serializer_error(ValidationError())) from original_exc

        return value

    def _coerce_tag_entry(self, data):
        entry_id = data['entry_id']
        tag_entry = RequestTagEntry.objects.filter(entry_id=entry_id).first()
        if tag_entry is None:
            raise ValidationError(detail='no tag entry with id {}'.format(entry_id))
        data['tag_entry'] = tag_entry


class RequestTagSerializer(ModelSerializer):
    entry_id = UUIDField()

    submitted_at = TimestampField()

    performer = RequestUserSerializer()

    tag_id = UUIDField()
    tag_type = CharField()
    tag_description = SerializerMethodField()

    request_id = CharField(allow_null=True)
    request_origin = RequestOriginField()

    original_user_trait = SerializerMethodField()
    original_user_id = UUIDField(allow_null=True)

    related_phone = PhoneNumberField(allow_null=True)
    related_user_id = UUIDField(allow_null=True)

    comment = CharField(allow_blank=True)

    class Meta:
        model = RequestTagEntry
        fields = [
            'entry_id',
            'submitted_at',
            'performer',
            'tag_id',
            'tag_type',
            'tag_description',
            'request_id',
            'request_origin',
            'original_user_trait',
            'original_user_id',
            'related_phone',
            'related_user_id',
            'comment',
        ]

    def get_original_user_trait(self, obj):
        return PhoneNumberField().to_representation(obj.original_phone)

    def get_tag_description(self, obj):
        tag_description_helper = self.context['tag_description_helper']

        tag_category = tag_description_helper.get_tag_category_entry(obj.tag_id, obj.tag_type)

        if obj.tag_type == RequestTagType.OLD.value:
            serializer = RequestTagCategorySerializer
        elif obj.tag_type == RequestTagType.NEW.value:
            serializer = RequestTreeTagCategorySerializer
        else:
            raise NotImplementedError

        formatted_tag_category = serializer(
            instance=tag_category, context={'tag_description_helper': tag_description_helper}
        ).data
        tag_description = formatted_tag_category['description']

        return tag_description
