import logging

from django.http.response import Http404
from rest_framework.response import Response

from cars.django.renderers import MP3Renderer

from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory, DriveRolePermissionFactory

from ..core.pagination_helper import CursorPagination, QuerySetWrapMixin
from ..serializers.common import EntriesListArgumentsSerializer, CallTrackArgumentsSerializer

LOGGER = logging.getLogger(__name__)


class PaginatedEntriesAPIView(DrivePermissionAPIView):
    arguments_serializer_class = EntriesListArgumentsSerializer

    pagination_class = CursorPagination

    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    request_helper = None
    get_data_method_name = 'get_entries'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._queryset_helper = QuerySetWrapMixin(
            helper_instance=self.request_helper,
            get_data_method_name=self.get_data_method_name
        )

    def get_queryset(self):
        self._queryset_helper.initialize(request=self.request, **self.request.arguments)
        return self._queryset_helper

    def do_get(self, request):
        data = self.paginate_queryset(self.get_queryset())
        response = self.get_paginated_response(data)
        return response


class CallTrackAPIView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_manager', 'support_py_calls'), require_all=False)]

    arguments_serializer_class = CallTrackArgumentsSerializer

    renderer_classes = [MP3Renderer]

    helper = None

    def do_get(self, request):
        track_or_external_call_id = request.arguments['track_id']

        try:
            external_call_id = int(track_or_external_call_id)
            data = self.helper.get_call_track(external_call_id=external_call_id)
        except (TypeError, ValueError):
            data = self.helper.get_call_track(track_id=track_or_external_call_id)

        if data is None:
            raise Http404('no call track entry with id provided: {}'.format(track_or_external_call_id))

        return Response(data)
