import json
import logging

from rest_framework.mixins import ListModelMixin
from rest_framework.response import Response
from rest_framework.status import HTTP_401_UNAUTHORIZED, HTTP_500_INTERNAL_SERVER_ERROR

from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory
from cars.django.views import CarsharingAPIView

from ..core import Chat2DeskWebHookHelper, SolomonHelper
from ..core.pagination_helper import ChatMessagesCursorPagination

from ..serializers.chat2desk import Chat2DeskMessagesListArgumentsSerializer, Chat2DeskMessagesModelSerializer

from ..models import Chat2DeskMessageEntry


LOGGER = logging.getLogger(__name__)


class Chat2DeskWebHookAPIView(CarsharingAPIView):
    solomon_helper = SolomonHelper('chat2desk_webhook')

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._helper = Chat2DeskWebHookHelper.from_settings()

    def do_post(self, request):
        if not self._helper.authorize_request(request):
            return Response(status=HTTP_401_UNAUTHORIZED)

        response_message = response_status = None
        data = request.data

        try:
            entry = json.loads(data) if isinstance(data, str) else data
            hook_type = self._helper.process_entry(entry)

        except Exception as exc:
            hook_type = 'error'
            response_message = 'error processing data: {}'.format(str(exc))
            response_status = HTTP_500_INTERNAL_SERVER_ERROR
            LOGGER.error(response_message)

        self.solomon_helper.increment_counter(hook_type)

        response = Response(response_message, status=response_status)
        return response


class Chat2DeskMessagesAPIView(ListModelMixin, DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    pagination_class = ChatMessagesCursorPagination

    serializer_class = Chat2DeskMessagesModelSerializer

    arguments_serializer_class = Chat2DeskMessagesListArgumentsSerializer

    def do_get(self, request, *args, **kwargs):
        # temporarily disable until roles are determined
        # from rest_framework.status import HTTP_404_NOT_FOUND
        # return Response(status=HTTP_404_NOT_FOUND)
        return self.list(request, *args, **kwargs)

    def get_queryset(self):
        return Chat2DeskMessageEntry.objects.prefetch_related('related_client', 'related_operator__staff_entry_binding')

    def filter_queryset(self, queryset):
        # consider processing dialog id directly or adding "or" statement
        queryset = queryset.filter(
            related_client=self.request.arguments['client_entry'],
            transport=self.request.arguments['transport'],
        )

        filters = {}

        if 'since' in self.request.arguments:
            filters['time_id__gte'] = self.request.arguments['since']

        if 'until' in self.request.arguments:
            filters['time_id__lt'] = self.request.arguments['until']

        if filters:
            queryset = queryset.filter(**filters)

        return queryset
