import logging

from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST, HTTP_405_METHOD_NOT_ALLOWED

from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory

from ..core.request_tags import RequestTagProcessingHelper, TagDescriptionHelper
from ..serializers.request_tags import (
    RequestTagCategoryListArgumentsSerializer,
    RequestTagListArgumentsSerializer,
    RequestTagAssignmentArgumentsSerializer,
    RequestTagRemovalArgumentsSerializer,
    RequestTagSerializer,
    RequestTagCategorySerializer,
    RequestTreeTagCategorySerializer,
    RequestTagType,
)
from ..models.call_tags import RequestTagEntry


LOGGER = logging.getLogger(__name__)


class RequestOriginView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    request_tag_helper = RequestTagProcessingHelper()

    def do_get(self, request):
        request_origins = self.request_tag_helper.get_all_request_origins()
        return Response(
            {
                'entries_count': len(request_origins),
                'data': request_origins,
            }
        )


class RequestTagCategoryView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    request_tag_helper = RequestTagProcessingHelper()
    request_tag_description_helper = TagDescriptionHelper()

    arguments_serializer_class = RequestTagCategoryListArgumentsSerializer

    def do_get(self, request):
        tag_categories = self.request_tag_helper.get_all_tag_categories(**request.arguments)

        if request.arguments['tag_type'] == RequestTagType.OLD:
            serializer = RequestTagCategorySerializer
        elif request.arguments['tag_type'] == RequestTagType.NEW:
            serializer = RequestTreeTagCategorySerializer
        else:
            raise NotImplementedError

        formatted_entries = [
            serializer(tag_category, context={'tag_description_helper': self.request_tag_description_helper}).data
            for tag_category in tag_categories
        ]

        formatted_entries.sort(key=lambda x: (x['order'], x['description']))  # description to be removed

        return Response(
            {
                'entries_count': len(formatted_entries),
                'data': formatted_entries,
            }
        )


class RequestTagView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    request_tag_description_helper = TagDescriptionHelper()

    arguments_serializer_class = RequestTagListArgumentsSerializer
    serializer_class = RequestTagSerializer

    def get_arguments_serializer_context(self):
        context = super().get_arguments_serializer_context()
        context['tag_description_helper'] = self.request_tag_description_helper
        return context

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['tag_description_helper'] = self.request_tag_description_helper
        return context

    def do_get(self, request):
        request_origin = request.arguments['request_origin']
        request_id, related_request_ids = request.arguments['request_id'], request.arguments['related_request_ids']

        entry_tags = RequestTagEntry.objects.filter(request_origin=request_origin.value)

        if related_request_ids:
            entry_tags = entry_tags.filter(request_id__in=related_request_ids)
        else:
            entry_tags = entry_tags.filter(request_id=request_id)

        data = self.get_serializer(entry_tags, many=True).data
        return Response({'tags': data})


class RequestTagAssignmentView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    request_tag_helper = RequestTagProcessingHelper()
    request_tag_description_helper = TagDescriptionHelper()

    arguments_serializer_class = RequestTagAssignmentArgumentsSerializer
    serializer_class = RequestTagSerializer

    def get_arguments_serializer_context(self):
        context = super().get_arguments_serializer_context()
        context['tag_description_helper'] = self.request_tag_description_helper
        return context

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['tag_description_helper'] = self.request_tag_description_helper
        return context

    def do_post(self, request):
        if request.arguments.get('tag_entry', None) is not None:
            return Response(
                {
                    'error': '"patch" method mut be used instead of "post"',
                },
                status=HTTP_405_METHOD_NOT_ALLOWED,
            )

        performed_by = request.user
        tag_entry = self.request_tag_helper.upsert_request_tag(performed_by=performed_by, **request.arguments)

        data = self.get_serializer(instance=tag_entry).data
        return Response({'entry': data})

    def do_put(self, request):
        error_message = None

        tag_entry = request.arguments.get('tag_entry', None)

        if tag_entry is None:
            error_message = 'entry_id.missing'
        else:
            if tag_entry.request_origin != request.arguments['request_origin'].value:
                error_message = 'request_origin.differ'
            elif tag_entry.request_id is not None and tag_entry.request_id != request.arguments['request_id']:
                error_message = 'request_id.differ'

        if error_message is not None:
            return Response({'error': error_message}, status=HTTP_400_BAD_REQUEST)

        performed_by = request.user
        tag_entry = self.request_tag_helper.upsert_request_tag(performed_by=performed_by, **request.arguments)

        data = self.get_serializer(instance=tag_entry).data
        return Response({'entry': data})


class RequestTagRemovalView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    request_tag_helper = RequestTagProcessingHelper()

    arguments_serializer_class = RequestTagRemovalArgumentsSerializer

    def do_post(self, request):
        performed_by = request.user
        tag_entry = request.arguments['tag_entry']
        self.request_tag_helper.delete_request_tag(tag_entry=tag_entry, performed_by=performed_by)
        return Response()
