import json
import logging

from django.db import transaction
from django.db import IntegrityError
from django.utils import timezone

from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.models.car_document import CarDocument, CarDocumentAssignment
from cars.service_app.models.history import DriveCarDocumentHistory, DriveCarDocumentAssignmentHistory
from cars.carsharing.models.car_hardware import (
    CarHardwareBeacon,
    CarHardwareHead,
    CarHardwareModem,
    CarHardwareSim,
    CarHardwareVega,
)
from cars.core.history import HistoryManager

LOGGER = logging.getLogger(__name__)


class AssemblyManager:

    def __init__(self, user):
        self._user = user
        self._documents_history = HistoryManager(DriveCarDocumentHistory)
        self._assignments_history = HistoryManager(DriveCarDocumentAssignmentHistory)

    class BeaconDoesNotExistError(Exception):
        pass

    class DeviceAssignedToDifferentCarError(Exception):

        def __init__(self, conflict_car, *args, **kwargs):
            super().__init__(*args, **kwargs)
            self.conflict_car = conflict_car

    class GenericDocumentIDParseError(Exception):
        pass

    class SimAlreadyAssignedError(Exception):
        pass

    class SimAlreadyPresentError(Exception):
        pass

    class SimDoesNotExistError(Exception):
        pass

    class VegaDoesNotExistError(Exception):
        pass

    @staticmethod
    def update_blob(document):
        def build_sim_blob(sim):
            if not sim:
                return None
            return {
                'icc': str(sim.icc) if sim.icc else None,
                'phone_number': str(sim.phone_number) if sim.phone_number else None,
            }

        def build_simple_json_blob(reg_document):
            result = {}
            for k, v in reg_document.__dict__.items():
                if k == 'id' or k.startswith('_'):
                    continue
                if not v:
                    result[k] = None
                else:
                    result[k] = str(v)
            return result

        blob = None
        impl = document.get_impl()
        if impl is None:
            return
        if document.type == 'car_hardware_sim':
            blob = build_sim_blob(impl)
        elif document.type == 'car_hardware_beacon':
            blob = {
                'serial_number': str(impl.serial_number) if impl.serial_number else None,
                'imei': str(impl.imei) if impl.imei else None,
                'sim': build_sim_blob(impl.sim),
            }
        elif document.type == 'car_hardware_modem':
            blob = {
                'sim': build_sim_blob(impl.sim),
            }
        elif document.type == 'car_hardware_vega':
            blob = {
                'imei': str(impl.imei) if impl.imei else None,
                'primary_sim': build_sim_blob(impl.primary_sim),
                'secondary_sim': build_sim_blob(impl.secondary_sim),
            }
        elif document.type == 'car_hardware_head':
            blob = {
                'head_id': str(impl.head_id) if impl.head_id else None,
                'device_id': str(impl.device_id) if impl.device_id else None,
            }
        elif document.type == 'car_registry_document':
            blob = build_simple_json_blob(impl)
        elif document.type == 'car_insurance_policy':
            blob = build_simple_json_blob(impl)
        if blob is not None:
            document.blob = json.dumps(blob, ensure_ascii=False)

    def normalize_icc(self, icc):
        # According to https://st.yandex-team.ru/DRIVEBACK-185
        icc = str(icc)
        if icc.startswith('8970199'):  # beeline
            icc = icc[:18]
        elif icc.startswith('8970102'):  # megafon
            icc = icc[:17]
        else:
            raise RuntimeError('unreachable with icc=%s', icc)
        return icc

    def add_sim_card(self, phone_number, icc):
        icc = self.normalize_icc(icc)
        try:
            with transaction.atomic():
                sim = CarHardwareSim(
                    phone_number=phone_number,
                    icc=icc,
                )
                sim.save()

                car_document = CarDocument(
                    type=CarDocument.Type.CAR_HARDWARE_SIM.value,
                    added_by=self._user,
                    added_at=timezone.now(),
                    car_hardware_sim=sim,
                )
                self.update_blob(car_document)
                car_document.save()

                self._documents_history.add_entry(car_document, str(self._user.id))

        except IntegrityError:
            raise self.SimAlreadyPresentError

        return sim

    def bulk_add_sim_cards(self, sim_cards):
        # get existing sim cards not to create twice
        existing_icc = set()
        existing_phone_numbers = set()

        existing_objects = CarHardwareSim.objects.all()
        for sim in existing_objects:
            existing_icc.add(str(sim.icc))
            existing_phone_numbers.add(str(sim.phone_number))

        sim_db_objs = []
        car_document_objs = []

        sim_cards_added = 0
        for card in sim_cards:
            phone_number = card['phone_number']
            try:
                icc = self.normalize_icc(card['icc'])
            except RuntimeError:
                LOGGER.exception('bad icc')
                continue

            if phone_number in existing_phone_numbers:
                LOGGER.info(
                    'rejecting sim card (%s, %s) because sim with such phone number already present',
                    str(phone_number),
                    str(icc),
                )
                continue

            if icc in existing_icc:
                LOGGER.info(
                    'rejecting sim card (%s, %s) because sim with such icc already present',
                    str(phone_number),
                    str(icc),
                )
                continue

            # from now on, there's a warranty that phone_number and icc are new
            sim = CarHardwareSim(
                phone_number=phone_number,
                icc=icc,
            )
            sim_db_objs.append(sim)

            car_document = CarDocument(
                type=CarDocument.Type.CAR_HARDWARE_SIM.value,
                added_by=self._user,
                added_at=timezone.now(),
                car_hardware_sim=sim,
            )
            self.update_blob(car_document)
            car_document_objs.append(car_document)

            # add new icc and phone number to avoid duplicates within one batch
            existing_icc.add(str(sim.icc))
            existing_phone_numbers.add(str(sim.phone_number))

            sim_cards_added += 1

        with transaction.atomic():
            CarHardwareSim.objects.bulk_create(sim_db_objs)
            for document in car_document_objs:
                self.update_blob(document)
            CarDocument.objects.bulk_create(car_document_objs)

        return sim_cards_added

    def get_sim_card(self, icc):
        try:
            sim_card = CarHardwareSim.objects.get(icc=self.normalize_icc(icc))
            return sim_card
        except CarHardwareSim.DoesNotExist:
            sim_card = self.add_sim_card('', self.normalize_icc(icc))
            return sim_card

    def get_vega(self, imei, create_if_missing=False):
        # new format
        primary_sim_icc = None
        secondary_sim_icc = None
        if imei.startswith('MT-32K LTE'):
            tokens = imei.split(';')
            imei = tokens[1]
            if len(tokens) == 4:
                primary_sim_icc = tokens[2]
                secondary_sim_icc = tokens[3]

        try:
            vega = CarHardwareVega.objects.get(imei=imei)
            return vega
        except CarHardwareVega.DoesNotExist:
            if not create_if_missing:
                raise self.VegaDoesNotExistError
            vega = self.add_or_update_vega(
                imei,
                primary_sim_icc=primary_sim_icc,
                secondary_sim_icc=secondary_sim_icc
            )
            return vega

    def get_beacon(self, imei, create_if_missing=False, device_code=None):
        try:
            beacon = CarHardwareBeacon.objects.get(imei=imei)
            return beacon
        except CarHardwareBeacon.DoesNotExist:
            if not create_if_missing:
                raise self.BeaconDoesNotExistError
            beacon = self.add_or_update_beacon(device_code)
            return beacon

    def add_or_update_beacon(self, imei, sim_icc=None):
        serial_number, actual_imei = imei.split(',')

        beacon = CarHardwareBeacon.objects.filter(imei=actual_imei).first()
        if beacon is None:
            with transaction.atomic():
                beacon = CarHardwareBeacon(
                    imei=actual_imei,
                    serial_number=serial_number,
                )
                beacon.save()

                car_document = CarDocument(
                    type=CarDocument.Type.CAR_HARDWARE_BEACON.value,
                    added_by=self._user,
                    added_at=timezone.now(),
                    car_hardware_beacon=beacon,
                )
                self.update_blob(car_document)
                car_document.save()

                self._documents_history.add_entry(car_document, str(self._user.id))

        beacon_document = CarDocument.objects.get(car_hardware_beacon=beacon)

        try:
            if sim_icc is not None:
                sim = self.get_sim_card(sim_icc)
                if sim != beacon.sim and sim.is_used():
                    raise self.SimAlreadyAssignedError
                beacon.sim = sim
            else:
                beacon.sim = None
            beacon.save()
            self.update_blob(beacon_document)
            beacon_document.save()
            self._documents_history.update_entry(beacon_document, str(self._user.id))
        except IntegrityError:
            raise self.SimAlreadyAssignedError

        return beacon

    def add_or_update_vega(self, imei, primary_sim_icc=None,
                           secondary_sim_icc=None):
        vega = CarHardwareVega.objects.filter(imei=imei).first()
        if vega is None:
            with transaction.atomic():
                vega = CarHardwareVega(
                    imei=imei,
                )
                vega.save()

                car_document = CarDocument(
                    type=CarDocument.Type.CAR_HARDWARE_VEGA.value,
                    added_by=self._user,
                    added_at=timezone.now(),
                    car_hardware_vega=vega,
                )
                self.update_blob(car_document)
                car_document.save()
                self._documents_history.add_entry(car_document, str(self._user.id))

        vega_document = CarDocument.objects.get(car_hardware_vega=vega)

        if primary_sim_icc is not None:
            primary_sim = self.get_sim_card(primary_sim_icc)
            if primary_sim not in (vega.primary_sim, vega.secondary_sim) and primary_sim.is_used():
                raise self.SimAlreadyAssignedError
        else:
            primary_sim = None

        if secondary_sim_icc is not None:
            secondary_sim = self.get_sim_card(secondary_sim_icc)
            if secondary_sim not in (vega.primary_sim, vega.secondary_sim) and secondary_sim.is_used():
                raise self.SimAlreadyAssignedError
        else:
            secondary_sim = None

        if primary_sim is not None and primary_sim == secondary_sim:
            raise self.SimAlreadyAssignedError

        try:
            vega.primary_sim = primary_sim
            vega.secondary_sim = secondary_sim
            vega.save()
            self.update_blob(vega_document)
            vega_document.save()

            self._documents_history.update_entry(vega_document, str(self._user.id))
        except IntegrityError:
            raise self.SimAlreadyAssignedError

        return vega

    def add_or_update_head(self, head_id, device_id=None):
        head = CarHardwareHead.objects.filter(head_id=head_id).first()
        if head is None:
            with transaction.atomic():
                head = CarHardwareHead(head_id=head_id, device_id=device_id)
                head.save()

                car_document = CarDocument(
                    type=CarDocument.Type.CAR_HARDWARE_HEAD.value,
                    added_by=self._user,
                    added_at=timezone.now(),
                    car_hardware_head=head,
                )
                self.update_blob(car_document)
                car_document.save()

                self._documents_history.add_entry(car_document, str(self._user.id))
        else:
            head.device_id = device_id
            head_document = CarDocument.objects.get(car_hardware_head=head)
            head.save()
            self.update_blob(head_document)
            head_document.save()
            self._documents_history.update_entry(head_document, str(self._user.id))

        return head

    def add_or_update_modem(self, sim_icc):
        modem = CarHardwareModem.objects.filter(sim__icc=sim_icc).first()
        if modem is None:
            with transaction.atomic():
                sim = self.get_sim_card(sim_icc)
                if sim.is_used():
                    raise self.SimAlreadyAssignedError
                modem = CarHardwareModem(sim=sim)
                modem.save()

                car_document = CarDocument(
                    type=CarDocument.Type.CAR_HARDWARE_MODEM.value,
                    added_by=self._user,
                    added_at=timezone.now(),
                    car_hardware_modem=modem,
                )
                self.update_blob(car_document)
                car_document.save()
                self._documents_history.add_entry(car_document, str(self._user.id))

        return modem

    def get_generic_device_document(self, device_code):
        device_code = str(device_code)

        # modem
        if device_code.startswith('89701') and 17 <= len(device_code) <= 20:
            modem = self.add_or_update_modem(sim_icc=device_code)
            return CarDocument.objects.get(car_hardware_modem=modem)

        # vega
        if 14 <= len(device_code) <= 15 or device_code.startswith('MT-32K LTE'):
            vega = self.get_vega(imei=device_code, create_if_missing=True)
            return CarDocument.objects.get(car_hardware_vega=vega)

        # beacon
        if device_code.startswith('SN'):
            serial_number, imei = device_code.split(',')
            beacon = self.get_beacon(imei=imei, create_if_missing=True, device_code=device_code)
            return CarDocument.objects.get(car_hardware_beacon=beacon)

        # head
        if len(device_code) == 12:
            head = self.add_or_update_head(head_id=device_code, device_id=None)
            return CarDocument.objects.get(car_hardware_head=head)

        # transponder
        device_code = device_code.strip()
        if len(device_code) == 35 or (len(device_code) == 16 and device_code.startswith('06')):
            return self.add_or_update_generic_doc('car_transponder', json.dumps({'code': device_code}))

        # car airport pass
        if len(device_code) == 10 and device_code[2] == '-':
            return self.add_or_update_generic_doc('car_airport_pass', json.dumps({'code': device_code}))

        # transponder_spb
        if len(device_code) == 19 and device_code.startswith('636287500000'):
            return self.add_or_update_generic_doc('car_transponder_spb', json.dumps({'code': device_code}))

        # airport pass spb
        if len(device_code) == 6:
            return self.add_or_update_generic_doc('car_airport_pass_spb', json.dumps({'code': device_code}, ensure_ascii=False))

        LOGGER.error('We don\'t know what is "{}"'.format(device_code))

        raise self.GenericDocumentIDParseError

    def add_or_update_generic_doc(self, type, blob):
        with transaction.atomic():
            doc = CarDocument.objects.filter(type=type, blob=blob).first()
            if doc is not None:
                return doc
            doc = CarDocument(
                type=type,
                blob=blob,
                added_at=timezone.now(),
            )
            doc.save()
            self._documents_history.add_entry(doc, str(self._user.id))
            return doc

    def attach_generic_device_to_car(self, device_code, car, is_force=False):
        device_document = self.get_generic_device_document(device_code)

        current_assignment = (
            CarDocumentAssignment.objects
            .filter(
                document=device_document,
                unassigned_at__isnull=True
            )
            .first()
        )

        if current_assignment is not None:
            if current_assignment.car_id == car.id:
                return
            if not is_force:
                raise self.DeviceAssignedToDifferentCarError(current_assignment.car)
            else:
                updater = CarUpdater(current_assignment.car)
                updater.unassign_document(current_assignment, self._user, history_manager=self._assignments_history)

        updater = CarUpdater(car)
        updater.assign_document(device_document, assigner_user=self._user, history_manager=self._assignments_history)
