from rest_framework.serializers import CharField, ModelSerializer, SerializerMethodField

from cars.django.serializers import BaseSerializer
from cars.carsharing.models import Car
from cars.carsharing.models.car_assembly import CarAssemblyPhoto
from cars.carsharing.models.car_assembly_stage import CarAssemblyStage
from cars.carsharing.models.car_document import CarDocumentAssignment, CarDocument
from cars.service_app.serializers.car_hardware import (
    CarHardwareBeaconSerializer,
    CarHardwareHeadSerializer,
    CarHardwareModemSerializer,
    CarHardwareVegaSerializer,
)


class CarChangeVINViewArgumentsSerializer(BaseSerializer):

    vin = CharField(required=True)


class CarCreateByVINViewArgumentsSerializer(BaseSerializer):

    model_code = CharField(required=True)
    vin = CharField(required=True)


class CarSetResponsiblePickerArgumentsSerializer(BaseSerializer):

    picker = CharField(required=True)


class CarAssemblyPhotoSerializer(ModelSerializer):

    class Meta:
        model = CarAssemblyPhoto
        fields = '__all__'


class CarAssemblyStageSerializer(ModelSerializer):

    photos = SerializerMethodField()

    class Meta:
        model = CarAssemblyStage
        fields = [
            'id',
            'name',
            'photos',
        ]

    def get_photos(self, obj):
        all_photos = self.context['photos_serialized']
        return [
            x for x in all_photos if x['stage'] == obj.id
        ]


class ServiceAppCarAssemblySerializer(ModelSerializer):

    beacon = SerializerMethodField()
    head = SerializerMethodField()
    modem = SerializerMethodField()
    vega = SerializerMethodField()

    class Meta:
        model = Car
        fields = [
            'id',
            'imei',
            'number',
            'vin',

            'beacon',
            'head',
            'modem',
            'vega',
        ]

    def get_beacon(self, obj):
        beacon = self._get_active_device_of_type(
            type_=CarDocument.Type.CAR_HARDWARE_BEACON.value,
            car=obj,
        )
        if beacon is None:
            return None
        return CarHardwareBeaconSerializer(beacon).data

    def get_head(self, obj):
        head = self._get_active_device_of_type(
            type_=CarDocument.Type.CAR_HARDWARE_HEAD.value,
            car=obj,
        )
        if head is None:
            return None
        return CarHardwareHeadSerializer(head).data

    def get_modem(self, obj):
        modem = self._get_active_device_of_type(
            type_=CarDocument.Type.CAR_HARDWARE_MODEM.value,
            car=obj,
        )
        if modem is None:
            return None
        return CarHardwareModemSerializer(modem).data

    def get_vega(self, obj):
        vega = self._get_active_device_of_type(
            type_=CarDocument.Type.CAR_HARDWARE_VEGA.value,
            car=obj,
        )
        if vega is None:
            return None
        return CarHardwareVegaSerializer(vega).data

    def _get_active_device_of_type(self, type_, car):
        document_assignment = (
            CarDocumentAssignment.objects
            .filter(
                car=car,
                unassigned_at__isnull=True,
                document__type=type_,
            )
            .first()
        )

        if document_assignment is None:
            return None

        return document_assignment.document.get_impl()
