import json
import time
from abc import abstractmethod

from django.shortcuts import get_object_or_404
from django.utils import timezone
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.carsharing.models import Car, CarModel
from cars.carsharing.models.car_hardware import (
    CarHardwareSim,
)
from cars.service_app.core.assembly_manager import AssemblyManager
from cars.service_app.serializers.assembly import ServiceAppCarAssemblySerializer
from .base import ServiceAppAPIView
from ..serializers.assembly import (
    CarChangeVINViewArgumentsSerializer,
    CarCreateByVINViewArgumentsSerializer,
    CarSetResponsiblePickerArgumentsSerializer,
)
from ..serializers.carsharing import ServiceAppCarSerializer
from ..serializers.car_hardware import (
    CarHardwareBeaconSerializer,
    CarHardwareSimSerializer,
    CarHardwareVegaSerializer,
    UploadCarHardwareBeaconViewArgumentsSerializer,
    UploadCarHardwareSimViewArgumentsSerializer,
    UploadCarHardwareVegaViewArgumentsSerializer,
    CarHardwareAttachToCarViewArgumentsSerializer,
)


class CarCreateByVINView(ServiceAppAPIView):

    arguments_serializer_class = CarCreateByVINViewArgumentsSerializer

    def do_post(self, request, *args, **kwargs):
        model = CarModel.objects.filter(code=request.arguments['model_code']).first()
        if model is None:
            return Response(
                {
                    'status': 'errors',
                    'errors': ['car.model.unknown'],
                },
                status=HTTP_400_BAD_REQUEST,
            )

        car = Car.objects.filter(vin=request.arguments['vin']).first()
        if car is not None:
            return Response(
                {
                    'status': 'errors',
                    'errors': ['car.exists'],
                },
                status=HTTP_400_BAD_REQUEST,
            )

        if request.arguments['vin'].startswith('#MjVin*'):
            request.arguments['vin'] = request.arguments['vin'][7:]

        car = Car(
            model=model,
            vin=request.arguments['vin'].upper(),
        )
        car.updated_at = timezone.now()
        car.update_timestamp = time.time()
        car.save()

        client = SaasDriveAdminClient.from_settings()
        default_tags_json = model.default_tags or '[]'
        default_tags = json.loads(default_tags_json)
        for default_tag_description in default_tags:
            client.add_car_tag(
                car_id=str(car.id),
                tag_name=default_tag_description['tag_name'],
                comment=default_tag_description['comment'],
                priority=default_tag_description['priority'],
            )

        return Response(
            {
                'status': 'success',
                'car': ServiceAppCarSerializer(car).data,
            },
        )


class CarSetResponsiblePicker(ServiceAppAPIView):

    arguments_serializer_class = CarSetResponsiblePickerArgumentsSerializer

    def do_post(self, request, car_id):  # pylint: disable=unused-argument
        car = Car.objects.filter(id=car_id).first()
        if car is None:
            return Response(
                {
                    'status': 'errors',
                    'errors': ['car.missing'],
                },
                status=HTTP_400_BAD_REQUEST,
            )

        car.responsible_picker = request.arguments['picker']
        car.updated_at = timezone.now()
        car.update_timestamp = time.time()
        car.save()

        return Response(
            {
                'status': 'success',
                'car': ServiceAppCarSerializer(car).data,
            }
        )


class CarChangeVINView(ServiceAppAPIView):

    arguments_serializer_class = CarChangeVINViewArgumentsSerializer

    def do_post(self, request, car_id):  # pylint: disable=unused-argument
        car = Car.objects.filter(id=car_id).first()
        if car is None:
            return Response(
                {
                    'status': 'errors',
                    'errors': ['car.missing'],
                },
                status=HTTP_400_BAD_REQUEST,
            )

        car.vin = request.arguments['vin'].upper()
        car.updated_at = timezone.now()
        car.update_timestamp = time.time()
        car.save()

        return Response(
            {
                'status': 'success',
                'car': ServiceAppCarSerializer(car).data,
            }
        )


class UploadEntityDetailsView(ServiceAppAPIView):

    arguments_serializer_class = None
    serializer_class = None

    def get_queryset(self):
        return CarHardwareSim.objects

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        manager = AssemblyManager(user=self.requester_user)
        try:
            obj = self._perform_manager_action(manager, request.arguments)
        except manager.SimAlreadyPresentError:
            return Response(
                {
                    'status': 'errors',
                    'errors': [
                        'assembly.sim.already_present'
                    ],
                },
                status=HTTP_400_BAD_REQUEST,
            )
        except manager.SimDoesNotExistError:
            return Response(
                {
                    'status': 'errors',
                    'errors': [
                        'assembly.sim.does_not_exist'
                    ],
                },
                status=HTTP_400_BAD_REQUEST,
            )
        except manager.SimAlreadyAssignedError:
            return Response(
                {
                    'status': 'errors',
                    'errors': [
                        'assembly.sim.already_assigned'
                    ]
                },
                status=HTTP_400_BAD_REQUEST,
            )

        return Response(
            {
                'status': 'success',
                'object': self.serializer_class(obj).data,
            }
        )

    @abstractmethod
    def _perform_manager_action(self, manager, arguments):
        raise NotImplementedError


class CarHardwareAttachToCarView(ServiceAppAPIView):

    arguments_serializer_class = CarHardwareAttachToCarViewArgumentsSerializer

    def do_post(self, request, car_id):
        car = Car.objects.filter(id=car_id).first()
        if car is None:
            return Response(
                {
                    'status': 'errors',
                    'errors': [
                        'assembly.car.does_not_exist'
                    ]
                },
                status=HTTP_400_BAD_REQUEST,
            )

        manager = AssemblyManager(user=self.requester_user)
        error = None
        conflict_car = None

        try:
            manager.attach_generic_device_to_car(request.arguments['device_code'], car, request.arguments.get('force'))
        except manager.SimDoesNotExistError:
            error = 'assembly.sim.does_not_exist'
        except manager.SimAlreadyAssignedError:
            error = 'assembly.sim.already_assigned'
        except manager.VegaDoesNotExistError:
            error = 'assembly.vega.does_not_exist'
        except manager.BeaconDoesNotExistError:
            error = 'assembly.beacon.does_not_exist'
        except manager.DeviceAssignedToDifferentCarError as e:
            error = 'assembly.device.assigned_to_different_car'
            conflict_car = e.conflict_car
        except manager.GenericDocumentIDParseError:
            error = 'assembly.device.cant_detect'

        if error:
            return Response(
                {
                    'status': 'errors',
                    'errors': [
                        error,
                    ],
                    'car': ServiceAppCarAssemblySerializer(car).data,
                    'conflict_car': ServiceAppCarAssemblySerializer(conflict_car).data,
                },
                status=HTTP_400_BAD_REQUEST,
            )

        return Response(
            {
                'status': 'success',
                'car': ServiceAppCarAssemblySerializer(car).data,
            }
        )


class CarAttachedDevicesView(ServiceAppAPIView):

    def do_get(self, request, car_id):  # pylint:disable=unused-argument
        car = get_object_or_404(Car, id=car_id)
        return Response(ServiceAppCarAssemblySerializer(car).data)


class UploadCarHardwareSimView(UploadEntityDetailsView):

    arguments_serializer_class = UploadCarHardwareSimViewArgumentsSerializer
    serializer_class = CarHardwareSimSerializer

    def _perform_manager_action(self, manager, arguments):
        return manager.add_sim_card(**arguments)


class UploadCarHardwareBeaconView(UploadEntityDetailsView):

    arguments_serializer_class = UploadCarHardwareBeaconViewArgumentsSerializer
    serializer_class = CarHardwareBeaconSerializer

    def _perform_manager_action(self, manager, arguments):
        return manager.add_or_update_beacon(**arguments)


class UploadCarHardwareVegaView(UploadEntityDetailsView):

    arguments_serializer_class = UploadCarHardwareVegaViewArgumentsSerializer
    serializer_class = CarHardwareVegaSerializer

    def _perform_manager_action(self, manager, arguments):
        return manager.add_or_update_vega(**arguments)
