import datetime
import decimal
import logging
import os
import sys
from cars.aggregator.static_data import clients
from cars.core.constants import AppPlatform
from .utils import get_secret, get_hex_secret, set_secret_default


IS_TESTS = False


ROOT_PATH = os.path.dirname(os.path.dirname(__file__))
LOCALE_PATH = os.path.join(ROOT_PATH, 'messages')


ADMIN = {
    # UID whitelist.
    'acl': {
        14113065,   # dbarbashin
        43300445,   # taran-ms
        50680745,   # todiane
        86063197,   # deep-core
        122159482,  # a.rznv
        164421795,  # bergusman
        204286059,  # yndx.raspopova
        235842166,  # zxqfd555
        248152061,  # dd2.0
        287808946,  # @kosolapova
        311753795,  # S.kovalev3
        381382836,  # e.gamayun
        401077842,  # therabbitinspace
        438352826,  # a.a.semchenko
        569008069,  # vlvlgoal
        571443797,  # fleethead

        # GeoSaas.
        92176534,   # sofya.nv
        159801225,  # s.v.shevtsov
        458114214,  # imorozov333

        # Ext QA. Free rides. added to rebuild
        132834224,  # dmiturich
        597481343,  # koval8492

        # Supports.
        809,        # masha
        7933221,    # aromensky (@judge)
        48128874,   # eka5745
        50455056,   # @vorfolomea
        83245874,   # @bogomolova2a
        87107723,   # trudenko911
        136262196,  # smirnov.kvik (@senqs)
        137472394,  # @katsur
        140441953,  # @kuzmiak
        229510704,  # @shulyaevan
        229938395,  # viktoriia.fomina (@vfomina)
        242304044,  # sulfur21
        251859805,  # @xingie
        258873780,  # @zionik
        319674899,  # meddeli (@meddelin)
        323367802,  # kazaripov (@zaripovka)
        361168445,  # @booka
        394827457,  # @lizusv
        513722297,  # wandercrack (@krestyannikan)
        524413628,  # @kateche
        540732692,  # vlad2010d2 (@vladzzz)
        571204362,  # @dariakorchaka

        # Chat supports.
        161163085,  # @ivalova
        367550129,  # @oddeye

        # Yandex call center.
        484473795,  # @e-nesterenko
        522291629,  # nivanovauafm (@ivanonad)

        # Yang.
        6043821,    # IrinkaSot (@irinkasot)
        45647354,   # selivanova-ns (@selivanova-ns)
        94605155,   # @alekseimoga
        272463242,  # eiden14 (@miranda)
        380660359,  # olga.b1708 (@bob1708)

        # Вега. Free rides.
        576597490,  # Саша
        588670672,  # Артём

        # Rommich.
        32126202,   # @rommich

        # More supports with IDM problems
        1120000000093374,   # @s-ivannikov
        1120000000077993,   # @huziev
        1120000000082288,   # @trushindima
        1120000000096644,   # @rudenok
        1120000000082942,   # @dkurochkin
        1120000000082294,   # @rodionovl
    },
    'internal_callcenter_acl': {
        38194648,   # Demonicada@yandex.ru
        42622286,   # pazhitei@yandex.ru
        51853602,   # golubinann@yandex.ru
        54102480,   # nad-kulueva@yandex.ru
        59750513,   # viksem89@yandex.ru
        94632459,   # anhema@yandex.ru
        128015295,  # sakovich.ea@yandex.ru
        130860066,  # lolasommer@yandex.ru
        150191253,  # juliakomarova2012@yandex.ru
        239412326,  # rockjopdjaz@yandex.ru
        330570880,  # zapretcod@yandex.ru
        413093918,  # burjurger1@yandex.ru
        441620865,  # fedorov.dv7@yandex.ru
        467313590,  # miss.moon-kiss@yandex.ru
        484473795,  # nesterenko.evgeny2018@yandex.ru
        592440075,  # zinchenkoanechka@yandex.ru
        593407394,  # ksanitabiz@yandex.ru
        593449668,  # alena.v.nikiforova@yandex.ru
        594601137,  # fatyanovaantonina@yandex.ru
        595136081,  # k5usha1618@yandex.ru
        595340426,  # tinoid73@yandex.ru
        595861989,  # anngst@yandex.ru
        597666256,  # ponomargal@yandex.ru
        598649417,  # elena.valuiskova@yandex.ru
        602581541,  # tatyanasushentseva@yandex.ru

        602773516,  # internal-cc-test
    },
    'documents': {
        'thumbnails': {
            'greater_side_to_fit': 400,
            'watermarker': {
                'base_watermark_size': 120,
                'watermark_visibility': 160,
                'font_size': 14,
            },
        },
        'images': {
            'watermarker': {
                'base_watermark_size': 600,
                'watermark_visibility': 100,
                'font_size': 80,
            },
        },
    },
    'transport': {
        'banned_car_ids': {
            '5367dd67-cdb9-4811-b101-8a67fae88b69',
        }
    },
    'porsche': {
        'user_role': None,
    },
    'push_client': {
        'filename': 'logs/admin.log',
    },
    'video_watermarker': {
        'max_video_width': 640,
        'max_video_height': 640,
        'base_watermark_size': 130,
        'base_visibility': 110,
        'font_size': 17,
    },
}


API = {
    'location_prefix': 'aggregator',
}


APPMETRICA = {
    'URL': 'https://api.appmetrica.yandex.ru',
    'APPLICATION_ID': 917433,
    'tvm2': {
        'host': 'https://tvm-api.yandex.net/',
        'source': 2000615,
        'destination': 2000337,
        'secret': get_secret('CARSHARING_FRONTEND_TVM2_SECRET', 'CARSHARING_FRONTEND_TVM2_SECRET'),
    },
}


BILLING = {
    'cards': {
        'fiscal_title': 'Яндекс.Драйв',
        'fiscal_nds': 'nds_18',
        'clear_delay': datetime.timedelta(days=2),
    },
    'push_client': {
        'filename': 'logs/billing.log',
    },
    'solomon': {
        'service': 'billing',
    },
    'payment_processor_tick_interval': '* * * * * *',
}


BLACKBOX = {
    'url': 'https://pass-test.yandex.ru/blackbox',  # may be either internal BB or external
    'url_external': 'https://pass-test.yandex.ru/blackbox',  # external BB (either testing or production)
    'default_tvm_section': 'tvm',  # used to control default tvm section from settings
    'tvm': {
        'host': 'https://tvm-api.yandex.net/',
        'source': 2000185,
        'destination': 222,  # external production blackbox
        'secret': get_secret('CARSHARING_TVM2_SECRET', 'CARSHARING_TVM2_SECRET'),
    },
    'db_fields_basic': [
        'account_info.reg_date.uid',
    ],
    'suid_weights': {
        # Important
        'subscription.suid.15': 2,  # 15 - Платёжный Паспорт (пароль)
        'subscription.suid.19': 2,  # 19 - Баланс-Билинг
        'subscription.suid.24': 2,  # 24 - РСЯ
        'subscription.suid.47': 2,  # 47 - webmaster
        'subscription.suid.48': 2,  # 48 - metrika
        'subscription.suid.52': 2,  # 52 - wikimaps (народная карта)
        'subscription.suid.53': 2,  # 53 - xml
        'subscription.suid.64': 2,  # 64 - cloudpaid
        'subscription.suid.91': 2,  # 91 - постофис (это админы какие-то будут)
        'subscription.suid.96': 2,  # 96 - Для пользователей "Охоты"
        'subscription.suid.110': 2,  # 110 - Я.Работа, Работодатели
        'subscription.suid.117': 2,  # 117 - подписка на сервис Телефонии
        # Not so important
        'subscription.suid.31': 1,  # 31 - calendar
        'subscription.suid.54': 1,  # 54 - music
        'subscription.suid.85': 1,  # 85 - Партнерский интерфейс
        'subscription.suid.97': 1,  # 97 - Яндекс.Авиабилеты
        'subscription.suid.98': 1,  # 98 - SID Кинопоиск
        'subscription.suid.114': 1,  # 114 - Пользовтаель Taxi
        'subscription.suid.116': 1,  # 116 - толока (не различает, исполнитель или заказчик)
    },
    'min_allowed_weight': 2,
}


CALCULATOR = {
    'client': {
        'class': 'cars.calculator.core.client.CalculatorClient',
        'root_url': os.environ.get('CARSHARING_CALCULATOR_ROOT_URL'),
        'retries': 1,
        'pool_maxsize': 8,
        'timeout': 2,
        'verify_ssl': True,
    },
}


CALLCENTER = {
    'acl': None,
    'call_registration_server_uid': 4009959856,  # drive-callcenter
    'user_link_ttl': 1200,  # in seconds
    'link_patterns': {
        'user_card': 'https://carsharing.yandex.ru/admin/#/call-center/{}/info/',
        'enter_phone': 'https://carsharing.yandex.ru/admin/#/call-center/{}/phone/',
    },
    'push_client': {
        'filename': 'logs/callcenter.log',
    },
    'base_url': 'https://tel.yandex-team.ru/robot.php/?',
    'api': {
        'client_class': 'unittest.mock.MagicMock',
        'url': '/mod.cipt-call-center/api',
        'stat_url': '/mod.cipt-stat/api',
        'source_routing_url': '/mod.cipt-call-center/API/CALLCENTER/SRCROUTING',
        'get_settings_url': '/mod.cipt-admin/api/routing/applications/get/{application_source}/CONTENT',
        'set_settings_url': '/mod.cipt-admin/api/routing/applications/update/{application_source}/CONTENT',
        'user': 'robot-carsharing',
        'secret_key': get_secret('CARSHARING_CALLCENTER_SECRET_KEY', 'CARSHARING_CALLCENTER_SECRET_KEY'),
    },
    'request_timeout': 30,  # in seconds; temporarily increase due to works
    'saas_history_token': get_secret('CARSHARING_ROBOT_CARSHARING_TOKEN', 'CARSHARING_ROBOT_CARSHARING_TOKEN'),
    'call_events_url': 'https://testing.carsharing.yandex.net/api/support_staff/call_event',
    'mds': {
        'access_key_id': get_secret(
            'CARSHARING_CALLCENTER_MDS_ACCESS_KEY_ID', 'CARSHARING_CALLCENTER_MDS_ACCESS_KEY_ID'
        ),
        'secret_access_key': get_secret(
            'CARSHARING_CALLCENTER_MDS_SECRET_ACCESS_KEY', 'CARSHARING_CALLCENTER_MDS_SECRET_ACCESS_KEY'
        ),
    },
    'default_performer_id': '74d9665e-04e4-4d83-b6f5-ae0b233b7c1e',  # robot-carsharing
    'distribute_apps': {
        'msk': 'CARSH_MOW_DIST_TEST',
        'spb': 'CARSH_TEST_DIST',
    },
    'tag_description_invalidation_enabled': True,
}


CALLCENTER_INTERNAL = {
    'acl': None,
    'role': 'internal_callcenter_operator',
}

CAR_DOCUMENTS = {
    'mds': {
        'mds_bucket_name': 'carsharing-car-documents',
    }
}

CARSHARING = {
    'acceptance': {
        'mds_bucket_name': 'carsharing-acceptances',
    },
    'reservation': {
        'constraints': [
            {
                'span': datetime.timedelta(hours=2),
                'count': 1,
            },
            {
                'span': datetime.timedelta(days=1),
                'count': 5,
            },
        ],
        'constrained_max_duration_seconds': 17,
        'full_max_duration_seconds': 20 * 60,
    },
    'plus': {
        'reservation_full_max_duration_seconds': 25 * 60,
        'extended_free_parking_start': datetime.time(hour=5, minute=30, second=0),
        'extended_free_parking_finish': datetime.time(hour=6, minute=0, second=0),
        'discount_multiplier': decimal.Decimal('0.95'),
    },
    'solomon': {
        'service': 'carsharing',
    },
    'tags': {
        'token': get_secret('CARSHARING_AUTO_TAGS_TOKEN', 'CARSHARING_AUTO_TAGS_TOKEN'),
    },
    'registry_manager': {
        'join_threads': False,
    }
}


COMMON = {
    'solomon': {
        'service': 'common',
    },
    'default_performer_id': '74d9665e-04e4-4d83-b6f5-ae0b233b7c1e',  # robot-carsharing
}


DB = {
    'default': {
        'host': None,
        'port': None,
        'name': None,
        'user': None,
        'password': None,
    },
}
DB_RO_ID = 'default'
DB_CC_ID = 'default'


DRIVE = {
    'intro': {
        'user_setup_class': 'cars.drive.core.user_setup.UserSetup',
    },
    'carsharing': {
        'car_list_location_updated_at_max_delta': datetime.timedelta(minutes=15),
        'car_list_since_threshold': datetime.timedelta(minutes=5),
    },
    'document_images': {
        'greater_side_to_fit': 500,
    },
    'early_access': {
        'enabled': True,
        'acl': {
            # Вега.
            576597490,  # Саша
            588670672,  # Артём

            506055427,  # kozhuhar.kirill.dev
            574906675,  # ya.kozhuhar.kirill
            595142979,  # drivetst (@dbarbashin)

            # Ext QA.
            132834224,  # dmiturich
            597481343,  # koval8492

            572287766,  # yndx-drive-apple
        },
    },
    'min_app_version': {
        AppPlatform.ANDROID: '1.0.7',
        AppPlatform.IOS: '105.0.0',
    },
    'min_app_build': {
        AppPlatform.ANDROID: '123',
        AppPlatform.IOS: '291',
    },
    'push_client': {
        'filename': 'logs/drive.log',
    },
}


DEPTRANS = {
    'token': get_secret('CARSHARING_DEPTRANS_TOKEN', 'CARSHARING_DEPTRANS_TOKEN'),
}


EKA = {
    'login': 'yandexdrive',
    'password': 'null',
    'settlement_account': None,

    'time_to_block_card': 20 * 60,  # 20 mins. should be given in seconds
    'time_to_scan_fuel': 30 * 60,  # check fuel level for 30 mins after refueling
    'refuel_time_threshold': 12 * 60 * 60,  # 12 hours between refuelings

    'refuel_threshold': 40.0,
    'consider_fueled_threshold': 60.0,

    'bonus_amount': decimal.Decimal('100.00'),
    'bonus_comment': 'Заправка автомобиля по топливной карте',

    'robot_user_id': '74d9665e-04e4-4d83-b6f5-ae0b233b7c1e',  # robot-carsharing
}


FINES = {
    'charge': True,
    'send_email': True,
    'send_push': False,
    'send_sms': True,
    'yandexoid_only': False,
    'drive_only': False,
    'charge_limit': 2500,
    'charge_time_limit': datetime.timedelta(days=30*3),
    'emails_only': None,
    'mds': {
        'mds_bucket_name': 'carsharing-violations',
        'images_access_host': 'https://carsharing-violations.s3.mdst.yandex.net/',
    },
    'koap_no_discount': [
        '12.9.6',
        '12.9.7',
        '12.12.3',
        '12.9ч.6',
        '12.9ч.7',
        '12.12ч.3',
    ],
    'force_discount': [
        '12.09.3',
    ],
    'saas_tracks_token': get_secret('CARSHARING_ROBOT_CARSHARING_TOKEN', 'CARSHARING_ROBOT_CARSHARING_TOKEN'),
    'saas_nearest_session_token': get_secret('CARSHARING_ROBOT_CARSHARING_TOKEN', 'CARSHARING_ROBOT_CARSHARING_TOKEN'),
    # 'emails_only': ['timlukin@yandex.ru'],
    'export_table': '//home/carsharing/testing/fines/export_ext',
    'yndx_fines': {
        'token': get_secret('CARSHARING_AUTOCODE_SECRET_ID', 'CARSHARING_YNDX_FINES_TOKEN'),
    },
}


# change tag evolution robot token - used by production daemons only
set_secret_default(
    'CARSHARING_ROBOT_DRV_REQ_TOKEN',
    'CARSHARING_ROBOT_DRV_REQ_TOKEN',
    get_secret('CARSHARING_ROBOT_CARSHARING_TOKEN', 'CARSHARING_ROBOT_CARSHARING_TOKEN')
)


INCIDENT = {
    'host': 'https://carsharing.yandex-team.ru',
    'evacuation': {
        'ticket_queue': {
            'Москва': {'queue': 'DRIVETEST', },
        },
        'template_name': {
            'Москва': 'MOSCOW_MAIL_TEXT_DESCRIPTION_TEMPLATE',
        },
    },
    'default_performer_id': '74d9665e-04e4-4d83-b6f5-ae0b233b7c1e',  # robot-carsharing
    'extra_performer_id': '4e8d389b-e9e4-4e05-be38-9d5f9ab4ea96',  # robot-drv-req
    'evolution_extra_robot_token': get_secret('CARSHARING_ROBOT_DRV_REQ_TOKEN', 'CARSHARING_ROBOT_DRV_REQ_TOKEN'),
}


LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'verbose': {
            'format': ('[%(asctime)s][%(levelname)s] %(name)s '
                       '%(filename)s:%(funcName)s:%(lineno)d | %(message)s'),
        },
        'simple': {
            'format': '%(levelname)s %(message)s'
        },
    },
    'handlers': {
        'console': {
            'level': 'DEBUG',
            'class': 'logging.StreamHandler',
            'formatter': 'verbose'
        },
    },
    'loggers': {
        '': {
            'handlers': ['console'],
            'level': 'INFO',
        },
        'botocore': {
            'level': 'WARNING',
            'propagate': True,
        },
        'cars.telematics': {
            'handlers': ['console'],
            'level': 'INFO',
            'propagate': False,
        },
        'PIL.Image': {
            'level': 'INFO',
            'propagate': True,
        },
        'planex': {
            'level': 'DEBUG',
            'handlers': ['console'],
            'propagate': False,
        },
        'pymorphy2.opencorpora_dict.wrapper': {
            'level': 'WARNING',
            'propagate': True,
        },
        'requests.packages.urllib3': {
            'level': 'DEBUG',
            'handlers': ['console'],
            'propagate': False,
        },
        's3transfer': {
            'level': 'INFO',
            'propagate': True,
        },
    }
}


CAR_INDEX = {
    'CACHE_TIMEOUT': 10,
    'DELAY': 1,  # 1s delay for SaaS replicas to become in-sync.
    'MAX_DELTA': 60 * 15,  # Limit delta requests to the last 15 minutes.
    'SAAS_CONFIG': {
        'index_host': 'saas-indexerproxy-maps-prestable.yandex.net',
        'search_host': 'saas-searchproxy-maps-prestable.yandex.net:17000',
        'key': 'd948371d599fb543eaae1c83a3e3e2df',
        'service_name': 'car_sharing_app',
        'timeout': 1,
    },
}

DJANGO_DEBUG = False


DJANGO_AUTHENTICATION_BACKENDS = [
    'cars.core.authorization.anonymous_auth_mechanism.Mechanism',
    # Disable BlackBox auth at all
    # 'django_yauth.authentication_mechanisms.oauth.Mechanism',
    # 'django_yauth.authentication_mechanisms.cookie.Mechanism',
]

DJANGO_MIDDLEWARE = [
    'cars.django.middleware.RequestMetaSetterMiddleware',
    'cars.django.middleware.SolomonMonitoringMiddleware',
    'cars.django.middleware.IDMxTVM2Middleware',
    'cars.aggregator.middleware.RequestCityMiddleware',
    'cars.aggregator.middleware.RequestClientMiddleware',
    'cars.aggregator.middleware.LocaleProxyMiddleware',
    'cars.users.middleware.YandexAuthenticationMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.locale.LocaleMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
]


DOCS_HASHES_KEY = get_hex_secret('CARSHARING_DOCS_HASHES_KEY', 'CARSHARING_DOCS_HASHES_KEY')


FEATURES = {
    'banned_operators': {},  # Client -> set<LocalOperator>
    'banned_models': {},     # Client -> set<CarModel>
    'scheme_only_delimobil_for_transport': False,
    'special_clients': {     # String code -> Client
        'powerup': clients.AGGREGATOR_PLUS,
    },
}


GEOCODER = {
    'url': 'http://geocode-net.int01e.tst.maps.yandex.ru/1.x/',
}


IDM = {
    'client_tvm_id': 2001600,
}


MAX_DISABLED_SHARD = (2 ** 16)  # Forcibly move 100% of users to new backend


MDS = {
    'client_class': 'cars.core.mds.client.MDSClient',
    'encrypted_client_class': 'cars.core.mds.client.EncryptedMDSClient',

    'access_key_id': get_secret('CARSHARING_MDS_ACCESS_KEY_ID', 'CARSHARING_MDS_ACCESS_KEY_ID'),
    'secret_access_key': get_secret('CARSHARING_MDS_SECRET_ACCESS_KEY', 'CARSHARING_MDS_SECRET_ACCESS_KEY'),
    'service_name': 's3',

    'endpoint_url': 'https://s3.mdst.yandex.net',
    'verify_ssl': True,

    'logfile': 'logs/mds.log',
}


NEW_BILLING = {
    'client_class': 'unittest.mock.MagicMock',
    'root_url': 'https://admin.carsharing.yandex.net/api/staff/billing/payment/',
    'link_card_url': 'https://testing.carsharing.yandex.net/api/yandex/card/',
    'default_url_args': {
        'service': 'drive-frontend',
        'version': 'v5.0.0',
    },
    'default_headers': {
        'AppBuild': '1000',
    },
    'author_user_id': '74d9665e-04e4-4d83-b6f5-ae0b233b7c1e',  # robot-carsharing
    'author_user_token': get_secret('CARSHARING_ROBOT_CARSHARING_TOKEN', 'CARSHARING_ROBOT_CARSHARING_TOKEN'),
}

OCTOPUS = {
    'client_class': 'cars.core.octopus.Octopus',
    'url': 'https://octopus-stats-test.paysys.yandex-team.ru:8804',
    'api_key': get_secret('CARSHARING_OCTOPUS_API_KEY', 'CARSHARING_OCTOPUS_API_KEY'),
    'verify_ssl': True,
}


ORDERS = {
    'chunked_payment_amount': decimal.Decimal('314.15'),
    'debt_retry_interval': datetime.timedelta(days=1),
    'debt_order_termination': {
        'delay': 180,  # 3min.
        'threshold': decimal.Decimal(1024),
    },
    'export': {
        'path': '//home/carsharing/dev/orders',
    },
    'preliminary_payments': {
        'amount': decimal.Decimal('271'),
        'n_orders_threshold': 10,
        'enable_fraction_above_threshold': 0.5,
        'enable_fraction_below_threshold': 0.85,
    },
    'push_client': {
        'filename': 'logs/orders.log',
    },
}


PASSPORT = {
    'consumer': 'carsharing',
    'url': 'https://passport-test-internal.yandex.ru',
    'timeout': 20,
    'verify_ssl': True,
    'session_params': {
        'retries': 0,
    },
}


PEDESTRIAN_ROUTER = {
    'SAAS_CONFIG': {
        'index_host': None,
        'search_host': 'saas-searchproxy-maps-prestable.yandex.net:17000',
        'key': None,
        'service_name': 'default-router',
        'timeout': 10,
    },
    'MAX_STRAIGHT_DISTANCE': 10000,  # Don't try to query SaaS router for >10km long routes.
    'TIE_RADIUS': 100,  # Find nearby graph edges in 100m radius.
    'WALK_SPEED': 1.1,  # m/s
}


PUSH_CLIENT = {
    'landing': {
        'filename': 'logs/landing.log',
    },
    'radar': {
        'filename': 'logs/radar.log',
    },
    'taxi': {
        'filename': 'logs/taxi.log',
    },
}


RADAR = {
    'remind_interval': 60 * 10,
    'location_debounce_distance': 0.1,  # Update alert location every 100m passed.
}


REFUEL = {
    'acl': None,
    'refuel_threshold': 75.0,
    'telematics_commands_delay': 3.0,
    'warming': {
        'delay': 5,  # delay before start and between retries
    },
    'photo_recognition': {
        # TOOD: need toloka_project_id
        'toloka_project_id': 11471,
        'toloka_template_pool_id': 69193,
        'yt_data_dir': '//home/extdata/carsharing/dev/refuel_recognition',
    },
    'refuel_extraction': {
        'yt_source_dir': '//logs/carsharing-telematics-events-log/1d',
        'yt_source_stream_dir': '//logs/carsharing-telematics-events-log/stream/5min',
        'yt_target_dir': '//home/extdata/carsharing/dev/fuel_level_logs',
    },
    'refuel_report_mailing_list': [
        'timlukin@yandex-team.ru',
        'kovalevich@yandex-team.ru',
        'bzhidovlenkov@yandex-team.ru',
        # 'skulik@yandex-team.ru',
        # 'aryazanov@yandex-team.ru',
        # 'adilova@yandex-team.ru',
    ],
    'mds': {
        'mds_bucket_name': 'carsharing-refuel-reports',
        'file_access_host': 'https://carsharing-refuel-reports.s3.mdst.yandex.net/',
    },

}


REGISTRATION = {
    # Users to be presented with a fake registration.
    'fake_uids': {
        572287766,  # yndx-drive-apple
    },
    'start_date': datetime.date(2017, 2, 1),  # Used to check user's age and driver experience.
    'min_age': datetime.timedelta(days=21*365),
    'min_driving_experience': datetime.timedelta(days=2*365),
    'octopus': {
        'call_check_lag': 10,  # skip call status check submitted less than "value" seconds before
        'call_timeout': 80,  # 80s of no progress.
        'script_path': os.path.join(ROOT_PATH, 'registration', 'core', 'resources', 'call.json'),
        'prompts': [
            {
                'question': 'question_prompt_{}'.format(i),
                'thanks': 'thanks_{}'.format(i),
            }
            for i in [4, 5, 6, 8]  # Male voices only.
        ]
    },
    'solomon': {
        'service': 'registration',
    },
    'taxi_cashback': {
        'accrual_end_time': None,
        'phones_table_path': (
            '//home/extdata/carsharing/dev/promo/registration-taxi-cashback/registered-users'
        ),
        'rides_faulty_tables_path': '//home/taxi-dwh/public/drive/',
        'rides_table_path': '//home/taxi-dwh/public/drive/user_taxi_orders',
        'cashback_proportion': 0.2,  # 20 % of taxi rides cost
        'max_cashback_value': 2000.0,  # no more than 2000 rubles
    },
    'trusted_countries': {
        'BLR',
        'RUS',
        'UKR',
        'РОССИЙСКАЯ ФЕДЕРАЦИЯ',  # old version
    },
    'untrusted_countries': {
        'ARM',
        'TJK',
        'UZB',
        'TKM',
        'KGZ',
    },
}


REGISTRATION_GEOZONES = [
    {
        'name': 'moscow',
        'min_lat': 54.081427,
        'min_lon': 34.820895,
        'max_lat': 56.999369,
        'max_lon': 40.646347,
    },
    {
        'name': 'spb',
        'min_lat': 59.569297,
        'min_lon': 28.872026,
        'max_lat': 60.345481,
        'max_lon': 31.281733,
    },
    {
        'name': 'kazan',
        'min_lat': 53.973023,
        'min_lon': 47.109506,
        'max_lat': 56.505500,
        'max_lon': 53.957784,
    },
    {
        'name': 'sochi',
        'min_lat': 43.141670,
        'min_lon': 36.988083,
        'max_lat': 45.389298,
        'max_lon': 42.658678,
    }
]


REGISTRATION_YANG = {
    'document_images': {
        'greater_side_to_fit': 1600,
        'watermarker': {
            'base_watermark_size': 300,
            'watermark_visibility': 70,
            'font_size': 30,
        },
    },
    'fetcher': {
        'assignment_recreation_limit': 1,
    },
    'verifier': {
        'class': 'cars.registration_yang.core.verifier.YangSecretIdVerifier',
        'base_url': 'https://yang.yandex-team.ru/api/v1/assignments/',
        'token': get_secret('CARSHARING_YANG_ROBOT_OAUTH_TOKEN', 'CARSHARING_YANG_ROBOT_OAUTH_TOKEN'),
        'verify_requests': True,
    },
    'yt_tables': {
        'assignments': '//home/extdata/carsharing/testing/yang/requests',
        'results': '//home/extdata/carsharing/testing/yang/results',
        'results_archive': '//home/extdata/carsharing/testing/yang/results_archive',
    },
    'creator': {
        'batch_size': 100,
        'wait_minutes_to_create_pool': 5,
    },
    'preview_user_uid': 4008757818,
    'video_watermarker': {
        'max_video_width': 640,
        'max_video_height': 640,
        'base_watermark_size': 130,
        'base_visibility': 110,
        'font_size': 17,
    },
}


REQANS_LOGGER = {
    'push_client': {
        'filename': 'logs/reqans.log',
    },
}


REQUEST_AGGREGATOR = {
    'common': {
        'max_raw_request_time_range': datetime.timedelta(hours=3),
        'max_staff_binding_request_time_range': datetime.timedelta(days=7),
        'default_performer_id': '74d9665e-04e4-4d83-b6f5-ae0b233b7c1e',  # robot-carsharing
    },
    'altay': {
        'request_timeout': 15,  # in seconds
        'robot_uid': 1120000000067712,  # robot-carsharing
    },
    'audiotele': {
        'ftp_connection': {
            'host': 'cc-datz.audiotele.ru',
            'port': 1221,
            'username': 'ydrived',
            'password': get_secret('CARSHARING_AUDIOTELE_FTP_PASSWORD', 'CARSHARING_AUDIOTELE_FTP_PASSWORD'),
        },
        'mds_connection': {
            'aes_key': get_hex_secret('CARSHARING_AUDIOTELE_TRACKS_AES_KEY', 'CARSHARING_AUDIOTELE_TRACKS_AES_KEY'),
            'bucket_name': 'carsharing-support-audiotele',
            'file_access_host': 'https://carsharing-support-audiotele.s3.mdst.yandex.net/',
            'encrypt': True,
        },
        'robot_username': 'robot-drv-req',
        'robot_auth_token': get_secret('CARSHARING_AUDIOTELE_AUTH_TOKEN', 'CARSHARING_AUDIOTELE_AUTH_TOKEN'),
        'token_salt': get_secret('CARSHARING_AUDIOTELE_TOKEN_SALT', 'CARSHARING_AUDIOTELE_TOKEN_SALT'),
        'request_username': 'yandexdrive',
        'request_password': get_secret(
            'CARSHARING_AUDIOTELE_BASIC_AUTH_PASSWORD', 'CARSHARING_AUDIOTELE_BASIC_AUTH_PASSWORD'
        ),
        'request_timeout': 30,  # in seconds
    },
    'beeper': {
        'ftp_connection': {
            'host': 'ftp.msk.beeper.ru',
            'port': 2222,
            'username': 'yandex_drive_ro',
            'password': get_secret('CARSHARING_BEEPER_FTP_PASSWORD', 'CARSHARING_BEEPER_FTP_PASSWORD'),
        },
        'mds_connection': {
            'aes_key': get_hex_secret('CARSHARING_BEEPER_TRACKS_AES_KEY', 'CARSHARING_BEEPER_TRACKS_AES_KEY'),
            'bucket_name': 'carsharing-support-beeper',
            'file_access_host': 'https://carsharing-support-beeper.s3.mdst.yandex.net/',
            'encrypt': True,
        },
    },
    'chat2desk': {
        'robot_username': 'robot-drv-req',
        'robot_auth_token': get_secret('CARSHARING_CHAT2DESK_AUTH_TOKEN', 'CARSHARING_CHAT2DESK_AUTH_TOKEN'),
        'api_token': get_secret('CARSHARING_CHAT2DESK_API_TOKEN', 'CARSHARING_CHAT2DESK_API_TOKEN'),
        'request_timeout': 15,  # in seconds
        'export_table': '//home/carsharing/testing/chat2desk/chats/export',
    },
    'callcenter': {
        'export': {
            'operators': {
                'table': '//home/carsharing/testing/callcenter_operators/export',
                'department_urls': [
                    'as_5706_dep73515',  # drive support
                    # previously: 'as_5706',  # support
                    'as_7861_dep79112',  # drive assessors
                    # previously: 'as_7861',  # assessors
                    'ext_6398',  # drive external
                    # previously: 'ext_6398_9239',  # beeper
                    # previously: 'ext_6398_dep59984',  # audiotele
                    'as_5706_dep81212',  # drive support for business
                ],
            },
            'support_user_available_info': {
                'table': '//home/carsharing/testing/support/support_user_available_info/export',
                'actions': ['support_user_available_info'],
                'page_limit_per_role': 5000 / 50,  # approx. 50 entries per page
                'specific_fields': ['uid', 'username'],
            },
            'roles': {
                'page_limit_per_role': 5000 / 50,  # approx. 50 entries per page
                'excluded_roles_table': '//home/carsharing/testing/support/user_roles/export',
                'excluded_roles_list': [
                    'default_user',
                    'user_fake_role_for_landing_business_class',
                    'landing_*',
                    'tarif_access_*',
                    'tariff_access_*',
                    'tariff_hourly_*',
                    'user_access_*',
                    'GR_default_user_*',
                ],
                'included_roles_table': '//home/carsharing/testing/user_roles/export',
                'included_roles_list': [
                    'interface-admin',
                    'interface-external-callcenter-operator',
                    'interface-internal-callcenter-operator',
                    'interface-polygon-editor',
                    'interface-senior-ext-callcenter-operator',
                    'interface-support-manager',
                    'interface-support-operator',
                    'interface-support-operator-DTP',
                    'interface-support-operator-duty',
                    'interface-support-operator-payment',
                    'interface-support-operator-reg',
                    'internal_callcenter_operator',
                    'active_cleaner',
                    'active_tech_support',
                    'backoffice',
                    'cleaner_Chisty_Drive',
                    'cleaner_Chisty_Drive_business',
                    'cleaner_Chisty_Drive_premium',
                    'cleaner_StyleD',
                    'drive_team_user',
                    'driving_to_city',
                    'fueling_manager',
                    'fueling_toplivo_v_bak',
                    'fueling_toplivo_v_bak_0-24',
                    'fueling_toplivo_v_bak_09-20',
                    'fueling_toplivo_v_bak_21-08',
                    'fueling_toplivo_v_bak_admin',
                    'installator',
                    'Major47_stuff',
                    'manager_auto',
                    'picker',
                    'tech_support',
                    'tsvetochny_staff',
                ],
            },
        },
        'call_track_view_actions': [
            'support_py_manager',
            'support_py_calls',
        ],
        'denied_operator_view_roles': [
            'team_adm_level2',
            'team_adm_level3',
        ],
        'denied_data_view_roles': [
            'team_adm_level2',
            'team_adm_level3',
        ],
        'yang_workers_table': '//home/carsharing/support/workers/userskills',
    },
    'monitoring': {
        'export': {
            'table': '//home/carsharing/testing/support/monitoring/export',
        }
    },
    'nextcontact': {
        'robot_username': 'robot-drv-req',
        'robot_auth_token': get_secret('CARSHARING_AUDIOTELE_AUTH_TOKEN', 'CARSHARING_AUDIOTELE_AUTH_TOKEN'),
        'token_salt': '34cfb531-2d25-4395-9441-981f30602953',
    },
    'tags': {
        'yql_import': {
            'token': get_secret('CARSHARING_YQL_TOKEN', 'CARSHARING_YQL_TOKEN'),
            'cluster': 'hahn',
            'table': '//home/toloka/yang/export/assignments/results_v56',
            'table_update_lag': datetime.timedelta(hours=2),
        },
        'internal_cc_carsharing_table': '//home/sprav/assay/dslatsis/CallCenter/moderation/drive_v2',
        'internal_cc_carsharing_vip_table': '//home/sprav/assay/garmagelm/drive_reason_vip',
        'internal_cc_outgoing_table': '//home/sprav/assay/garmagelm/CallCenter/drive_outgoing',
        'audiotele_incoming_table': '//home/sprav/assay/garmagelm/drive/drive_outsourcing_reason',
        'audiotele_outgoing_table': '//home/sprav/assay/garmagelm/drive_outgoing/drive_outgoing_out',
        'export': {
            'common_table': '//home/carsharing/testing/support/request_tags/export',
            'cc_internal_carsharing_table': '//home/carsharing/testing/support/request_tags/cc_internal_carsharing_export',
            'cc_internal_carsharing_vip_table': '//home/carsharing/testing/support/request_tags/cc_internal_carsharing_vip_export',
            'cc_internal_altay_outgoing_table': '//home/carsharing/testing/support/request_tags/cc_internal_altay_outgoing_export',
            'cc_audiotele_incoming_table': '//home/carsharing/testing/support/request_tags/cc_audiotele_incoming_export',
            'cc_audiotele_outgoing_table': '//home/carsharing/testing/support/request_tags/cc_audiotele_outgoing_export',
        },
    },
    'call_data': {
        'export': {
            'cc_internal_incoming_table': '//home/carsharing/testing/support/call_data/cc_internal_incoming_export',
            'cc_internal_outgoing_table': '//home/carsharing/testing/support/call_data/cc_internal_outgoing_export',
            'cc_internal_routing_table': '//home/carsharing/testing/support/call_data/cc_internal_routing_export',
            'cc_audiotele_table': '//home/carsharing/testing/support/call_data/cc_audiotele_export',
            'cc_audiotele_track_table': '//home/carsharing/testing/support/call_data/cc_audiotele_track_export',
        },
    },
}


SAAS_DRIVE = {
    'url': 'https://testing.carsharing.yandex.net',
    'prestable': True,
    'version': None,
    'service': 'drive-frontend',
    'timeout': 10,
    'default_headers': {
        'AppBuild': '1000',
    },
    'token': get_secret('CARSHARING_AUTO_TAGS_TOKEN', 'CARSHARING_AUTO_TAGS_TOKEN'),
}

SEND_PUSH = {
    'mds': {
        'mds_bucket_name': 'carsharing-push-reports',
        'file_access_host': 'https://carsharing-push-reports.s3.mdst.yandex.net/',
    },
    'reporter': {
        'client_class': 'unittest.mock.MagicMock',
    }
}

SENTRY = {
    'feeds': None,
}


FEEDS = {
    'solomon': {
        'project': 'extmaps',
        'service': 'carsharing',
        'cluster': 'carsharing_dev',
    }
}


SEGMENTS = {
    'premium': {
        'tag': 'premium_user',
    },
    'tags_for_models': {
        'porsche_macan': 'premium_user',
        'porsche_carrera': 'premium_user',
    }
}


SERVICE_APP = {
    'roles': {
        'cleaner': {
            598310175,  # Operator1glow
            598312253,  # Operator2glow
            600860032,  # zxqfd-cleaner
            611415545,  # zxqfd-cleaner01
            611415303,  # zxqfd-cleaner02
            611415078,  # zxqfd-cleaner03
            614024083,  # drive.android.dev

            601230449,  # glowlab1
            601231182,  # glowlab2
            601231689,  # glowlab3
            601232044,  # glowlab4
            601232391,  # glowlab5
            601233039,  # glowlab6
            601233392,  # glowlab7
            603849433,  # glowlab9
            603863609,  # glowlab11
            603865304,  # glowlab12
            603866271,  # glowlab13
            609650719,  # glowlab14
            603868059,  # glowlab1010
            609355225,  # Glowlab15
            609355443,  # Glowlab16
            609355831,  # Glowlab17
            609356129,  # Glowlab18
            609650995,  # glowlab19
            609651487,  # glowlab20
            609651963,  # glowlab21
            615799676,  # glowlab22
            615800709,  # glowlab23
            615801436,  # glowlab24
            615802278,  # glowlab25

            578808997,  # Romanky4ma from toplivovbak, temporary access

            # Express cleaning from user_roles table
            14113065,  # keksite
            40958135,  # rodionov-leonid
            45344848,  # osmans50
            54062243,  # denissavinkov
            73630347,  # amirov-maxim
            92202361,  # egorahc
            171406314,  # uid-tr74hkne
            314233746,  # serdarovez
            410308036,  # jyniki
            418112573,  # esergikov1989
            432028769,  # agaevksh
            476662089,  # uid-phz6bf22
            551371901,  # uid-e3x6cmiu
            571005773,  # zasolockii
            573462135,  # uid-xv5mxbvs
            576306349,  # anastacia.aristarhova
            577431506,  # uid-yx3bbrvv
            580815551,  # alanramonov21
            590242025,  # uid-u7e2u2vy
            600860032,  # zxqfd-cleaner
            614024083,  # drive-android-dev
            626586199,  # ydtest-express-cleaner01
            627742093,  # uid-fphoiaoq
            633704749,  # krbim4anin140
            633918365,  # uid-shzg3nkx
            635763974,  # ibraimovazamat
            636525431,  # perfectlui
            636475215,  # ydtest-picker01
            1120000000070876,  # zomb-drive-minsk
            1130000021950335,  # lugovoy@suprotec.ru
        },
        'supervisor': {
            14113065,  # dbarbashin
            50680745,  # todiane
            86063197,  # deep-core
            122159482,  # a.rznv
            164421795,  # bergusman
            235842166,  # zxqfd555
            248152061,  # dd2.0
            571443797,  # fleethead

            # GeoSaaS
            92176534,  # sofya.nv
            159801225,  # s.v.shevtsov
            458114214,  # imorozov333
        },
        'technician': {
            600858420,  # zxqfd-technician
            574906675,  # ya.kozhuhar.kirill
            43300445,  # taran-ms

            40958135,  # rodionov-leonid
            103633586,  # Bodrikow
            204332413,  # kenny091993
            287808946,  # malish.kukush (@kosolapova)
            290018895,  # petrik2411
            582607497,  # karandashovvg
            588436182,  # Jeep664
            594685341,  # levshineugene
            601009624,  # loba4ewwww
            604627139,  # draiv.efim
            604663243,  # stanshishkov
            611647945,  # shishkovstan
            634233456,  # arsene.agadzhanyan

            # Ya.Auto
            95506,  # neosphere
            140380931,  # denis.itchenko
            503914227,  # ekolesni4enko
        },
    },
    'push_client': {
        'filename': 'logs/service_app.log',
    },
    'assembly': {
        'photos': {
            'mds_bucket_name': 'carsharing-assembly',
            'images_access_host': 'https://carsharing-assembly.s3.mdst.yandex.net/',
            'thumbnail_greater_side': 400,
        },
    },
    'tags': {
        'photos': {
            'mds_bucket_name': 'carsharing-tags',
            'images_access_host': 'https://carsharing-tags.s3.mdst.yandex.net/',
            'thumbnail_greater_side': 400,
        },
    },
    'min_app_build': 52,
}


SOLOMON = {
    'url': 'http://solomon.yandex.net',
    'project': 'carsharing',
    'service': 'api',
    'cluster': 'dev',
    'push_interval': 15,
    'token': get_secret('CARSHARING_STARTREK_TOKEN', 'CARSHARING_STARTREK_TOKEN'),
}

STAFF = {
    'base_url': 'https://staff-api.yandex-team.ru/v3',
    'persons_api_url': '/persons?',
    'name_locale': 'ru',
    'token': get_secret('CARSHARING_STAFF_TOKEN', 'CARSHARING_STAFF_TOKEN'),
    'fields': [
        'uid', 'login', 'name',
        'phones', 'work_phone', 'work_email',
        'is_deleted', 'official.is_dismissed', 'official.quit_at',
        'department_group.url',
    ],
    'request_timeout': 5,  # in seconds
    'request_limit': 250,
    'cache_invalidation_enabled': True,
}


STARTREK = {
    'token': get_secret('CARSHARING_STARTREK_TOKEN', 'CARSHARING_STARTREK_TOKEN'),
    'api_url': 'https://st-api.test.yandex-team.ru',
    'web_url': 'https://st.test.yandex-team.ru',
}


TAXI = {
    'corp_client': {
        'base_url': 'https://corp-client.taxi.tst.yandex.ru',
        'client_id': '05e7f8347bd840769520fb4bf2e582e5',
        'verify_ssl_cert': False,
        'sessionid2_cookie': ('3:1496843616.5.0.1496843616905:xr1Fzfm99gkENQAAuAYCKg'
                              ':8b.1|4006777567.0.2|278873.883224.waKsJiYlBvcVucPC2FViRd8JtFI'),
    },
}


TELEMATICS = {
    'api': {
        # Sort of service discovery.
        'backends': [
            'http://man1-1811.search.yandex.net:24451',
            'http://man1-7880.search.yandex.net:24451',
            'http://man2-1231.search.yandex.net:24451',
            'http://man2-1258.search.yandex.net:24451',
            'http://sas1-0359.search.yandex.net:24451',
            'http://sas1-4523.search.yandex.net:24451',
            'http://sas1-5337.search.yandex.net:24451',
            'http://sas1-5684.search.yandex.net:24451',
            'http://vla1-2021.search.yandex.net:24451',
            'http://vla1-2173.search.yandex.net:24451',
            'http://vla1-3021.search.yandex.net:24451',
            'http://vla1-4365.search.yandex.net:24451',
        ],
        'retries': 2,
        'timeout': 30.0,
    },
    'frontend': {
        'timeout': 25.0,
    },
    'packet_handlers': {
        'signal_collector': {
            'timeout': 2,
            'url': None,
        },
        'saas_submission': {
            # Service in SaaS, used for car lookup.
            'drive_service': {
                'index_host': 'saas-indexerproxy-maps-prestable.yandex.net',
                'key': '0e7d0426f506f52e7598c1b83cd6d17e',
                'service_name': 'drive',
                'timeout': 1,
            },
            # Service in SaaS, used for LK and drive quality monitor.
            'drive_navigator_service_prestable': {
                'index_host': 'saas-indexerproxy-maps-prestable.yandex.net',
                'key': '8a0760ca952635327da5df71730da3fd',
                'service_name': 'drive_navigator',
                'timeout': 1,
            },
            'drive_navigator_service_stable': {
                'index_host': 'saas-indexerproxy-maps.yandex.net',
                'key': '8a0760ca952635327da5df71730da3fd',
                'service_name': 'drive_navigator',
                'timeout': 1,
            },
            'status_cache': {
                'cached_state_ttl': 10,
                'get_state_url': 'http://localhost:8000/api/admin/v1/car-info-by-imei/status/',
            },
            'ride_cache': {
                'cached_state_ttl': 45,
                'get_state_url': 'http://localhost:8000/api/admin/v1/car-info-by-imei/ride/',
            },
        },
    },
    'proxy_class': 'cars.carsharing.core.telematics_proxy.TelematicsProxy',
    'push_client': {
        'filename': 'logs/telematics.log',
    },
    'solomon': {
        'service': 'telematics',
    },
}


THIRD_PARTY_APIS = {
    'autocode': {
        'system_id': 'yandex',
        'secret_id': get_secret('CARSHARING_AUTOCODE_SECRET_ID', 'CARSHARING_AUTOCODE_SECRET_ID'),
        'logs_table': None,
    },
    'renaissance': {
        'endpoint_url': 'http://stg.services.renins.com/Carsharing/SaveTrip',
        'partner_agreement_num': '001AT-17/37995',
    },
}


TRUST = {
    'client_class': 'cars.core.trust.TrustClient',
    'service_token': get_secret('CARSHARING_TRUST_SERVICE_TOKEN', 'CARSHARING_TRUST_SERVICE_TOKEN'),
    'root_url': 'https://trust-payments-test.paysys.yandex.net:8028',
    'default_product_id': '1',
    'timeout': 15,
    'verify_ssl': True,
    'back_url': None,
}


USERS = {
    'documents': {
        'datasync': {
            'host': 'http://api-stable.dst.yandex.net:8080',
            'endpoints_path': '/v1/personality/profile/persdata/',
            'verify_requests': False,
            'push_client': {
                'filename': 'logs/datasync.log',
            },
            'tvm2': {
                'host': 'https://tvm-api.yandex.net/',
                'source': 2000184,
                'destination': 2000060,
                'secret': get_secret('CARSHARING_TVM2_SECRET', 'CARSHARING_TVM2_SECRET'),
            },
        },
        'mds': {
            'aes_key': get_hex_secret('CARSHARING_DOCS_AES_KEY', 'CARSHARING_DOCS_AES_KEY'),
            'bucket_name': 'carsharing-documents',
        },
    },
    'export_table': '//home/extdata/carsharing/dev/users/export',
    'fast_export_table': '//home/extdata/carsharing/dev/users/fast_export',
    'neutral_tags': {
        'silver_client',
        'gold_client',
        'platinum_client',
        'blocked_duplicate_namber_pass',
        'blocked_duplicate_driver_license',
        'asked_to_delete',
        'old_license',
        'yandex_staff',
        'yandex_stuff',
    },
    'view_restrictions': {
        'deleting': {
            'denied_roles': [
                'team_adm_level2',
                'team_adm_level3',
            ],
            'required_user_tags': [
                'blocked_to_delete',
            ],
        },
        'blackbox_info': {
            'denied_roles': [
                'team_adm_level1',
                'team_adm_level2',
                'team_adm_level3',
            ],
        },
    },
}


XIVA = {
    'client_class': 'cars.core.pusher.Xiva',
    'url': 'http://push-sandbox.yandex.ru',
    'service': 'carsharing',
    'send_token': get_secret('CARSHARING_XIVA_SEND_TOKEN', 'CARSHARING_XIVA_SEND_TOKEN'),
    'subscription_token': get_secret('CARSHARING_XIVA_SUBSCRIPTION_TOKEN', 'CARSHARING_XIVA_SUBSCRIPTION_TOKEN'),
    AppPlatform.ANDROID: {
        'app_name': 'com.yandex.mobile.drive',
    },
    AppPlatform.IOS: {
        'app_name': 'ru.yandex.mobile.drive.inhouse',
    },
    'push_client': {
        'filename': 'logs/xiva.log',
    },
    'solomon': {
        'service': 'xiva',
    },
}


YASMS = {
    'client_class': 'kubiki.yasms.YaSms',
    'url': 'http://phone-passport-test.yandex.ru',
    'sender': 'carsharing',
    'sendsms': {
        'route': 'drive',
    },
}


SENDER = {
    'client_class': 'cars.core.sender.SenderClientCopyToMailing',
    'url_template': 'https://test.sender.yandex-team.ru/api/0/yandex.carshare/'
                    'transactional/{campaign}/send',
    'token': get_secret('CARSHARING_SENDER_TOKEN', 'CARSHARING_SENDER_TOKEN'),
}


TOLOKA = {
    'client_class': 'cars.ext.toloka.api.Client',
    'url': 'https://sandbox.toloka.yandex.ru',
    'token': get_secret('CARSHARING_TOLOKA_TOKEN', 'CARSHARING_TOLOKA_TOKEN'),
}


YAUTH_TEST_USER = None


YAUTH_MECHANISMS = [
    'cars.core.authorization.anonymous_auth_mechanism',
    # Disable BlackBox auth at all
    # 'django_yauth.authentication_mechanisms.cookie',
    # 'django_yauth.authentication_mechanisms.oauth',
]


YT = {
    'data': {
        'proxy': 'hahn',
        'token': get_secret('CARSHARING_YT_TOKEN', 'CARSHARING_YT_TOKEN'),
        'base_dir': '//home/extdata/carsharing/dev',
        'logs_dir': '//logs/extmaps-carsharing-backend-log/1d',
    },
    'locks': {
        'proxy': 'locke',
        'token': get_secret('CARSHARING_YT_TOKEN', 'CARSHARING_YT_TOKEN'),
        'base_dir': '//home/extmaps/carsharing/dev',
        'ttl': 10,
    },
}


ZOOKEEPER = {
    'hosts': [
        'man1-4702.search.yandex.net:2181',
        'myt1-1231.search.yandex.net:2181',
        'sas1-9217.search.yandex.net:2181',
        'myt1-1992.search.yandex.net:2181',
        'sas1-6533.search.yandex.net:2181',
    ],
    'workdir': None,
}


SOCIAL = {
    'url': 'https://api.social-test.yandex.ru',
    'profile_list_url': '/api/profiles',
    'consumer': 'carsharing-testing',
    'tvm2': {
        'host': 'https://tvm-api.yandex.net/',
        'source': 2000184,
        'destination': 2000381,
        'secret': get_secret('CARSHARING_SOCIAL_TVM2_SECRET', 'CARSHARING_SOCIAL_TVM2_SECRET'),
    },
    'timeout': 1,
}


ANIMALS = {
    'host': 'animals.search.yandex.net',
    'port': 4004,
    'auth': {
        'login': get_secret('CARSHARING_ANIMALS_TOKEN', 'CARSHARING_ANIMALS_TOKEN_LOGIN'),
        'token': get_secret('CARSHARING_ANIMALS_TOKEN', 'CARSHARING_ANIMALS_TOKEN'),
    },
}


EXPORT = {
    'tmp_path': '//home/carsharing/dev/data/user/_tmp',
    'temp_path': '//home/carsharing/dev/temp',
    'locks_config': {
        'proxy': 'hahn',
        'token': get_secret('CARSHARING_YT_TOKEN', 'CARSHARING_YT_TOKEN'),
        'base_dir': '//home/carsharing/dev',
        'ttl': 10,
    },
    'solomon_project': 'carsharing_analytics',
    'robot_oauth_token': get_secret('CARSHARING_ROBOT_CARSHARING_TOKEN', 'CARSHARING_ROBOT_CARSHARING_TOKEN'),
    'home_directory': '//home/carsharing/dev',
    # Sources
    'crypta_table': '//home/crypta/production/state/graph/v2/matching/by_id/mm_device_id/crypta_id',
    'apps_table_format': '//home/extdata/release/dashboard_carsharing/apps/{}',
    'crypta_gaid_table': '//home/crypta/production/state/graph/v2/matching/by_id/crypta_id/gaid',
    'crypta_yandexuid_table': '//home/crypta/production/state/graph/v2/matching/by_id/crypta_id/yandexuid',
    'crypta_profiles_table': '//home/crypta/production/profiles/export/profiles_for_14days',
    'current_score_table': '//home/carsharing/production/projects/DRIVE_211/result_log',
    'plus_subscribers_table': '//home/plus_data/share/subscribers_data_new',
    'security_issues_list_url': '/v2/issues?filter=queue:DRIVESECURITY&perPage=10000',
    'security_issue_url_format': '/v2/issues/{}',
    'social_profiles_url_format': '/api/profiles?uids={}',
    # Parts
    'users_table_part': '//home/carsharing/dev/data/user/_parts/users_table',
    'orders_table_part': '//home/carsharing/dev/data/user/_parts/orders_table',
    'bonus_account_table_part': '//home/carsharing/dev/data/user/_parts/bonus_account_table',
    'autocode_fine_table_part': '//home/carsharing/dev/data/user/_parts/autocode_fine_table',
    'billing_tasks_table_part': '//home/carsharing/dev/data/user/_parts/billing_tasks_table',
    'datasync_part': '//home/carsharing/dev/data/user/_parts/datasync',
    'security_part': '//home/carsharing/dev/data/user/_parts/security',
    'user_profiles_part': '//home/carsharing/dev/data/user/_parts/user_profiles',
    'users_apps_part': '//home/carsharing/dev/data/user/_parts/users_apps',
    'users_crypta_part': '//home/carsharing/dev/data/user/_parts/users_crypta',
    'users_crypta_profiles_part': '//home/carsharing/dev/data/user/_parts/users_crypta_profiles',
    'current_score_part': '//home/carsharing/dev/data/user/_parts/current_score',
    'plus_subscribers_part': '//home/carsharing/dev/data/user/_parts/plus_subscribers',
    'license_loss_part': '//home/carsharing/dev/data/user/_parts/license_loss',
    'plus_part': '//home/carsharing/dev/data/user/_parts/plus',
    'bin_attr_part': '//home/carsharing/dev/data/user/_parts/bin_attr',
    'users_bin_attr_part': '//home/carsharing/dev/data/user/_parts/users_bin_attr',
    # Tables
    'users_table': '//home/carsharing/dev/data/user/user_attributes',
    'orders_table': '//home/carsharing/dev/data/orders',
    # Global Export Tables
    'table_refund_issues_history': '//home/carsharing/dev/database/refund_issues_history',
}

THIEF_SAAS = {
    'prestable_host': 'saas-indexerproxy-maps-prestable.yandex.net',
    'host': 'saas-indexerproxy-maps-kv.yandex.net',
    'service': 'ac6ac7f7a6544f336202c0b058104374',
}
