import logging

import requests

import kubiki.geocoder

import cars.settings
from .urls import TaxiClientUrls


LOGGER = logging.getLogger(__name__)


class TaxiClient(object):

    def __init__(self, base_url, client_id, sessionid2_cookie, push_client, verify_ssl_cert=True):
        self._geocoder = kubiki.geocoder.Geocoder(url=cars.settings.GEOCODER['url'])
        self._urls = TaxiClientUrls(base_url=base_url, client_id=client_id)
        self._sessionid2_cookie = sessionid2_cookie
        self._verify_ssl_cert = verify_ssl_cert
        self._push_client = push_client

    def _request(self, *, method='GET', url, data=None):
        cookies = {
            'sessionid2': self._sessionid2_cookie,
        }

        try:
            response = requests.request(
                method=method,
                url=url,
                json=data,
                cookies=cookies,
                verify=self._verify_ssl_cert,
            )
        except Exception:
            LOGGER.exception('Taxi API request failed')
            response = None

        self._log_api_request(
            method=method,
            url=url,
            data=data,
            response=response,
        )

        return response

    def _log_api_request(self, method, url, data, response):
        log_data = {
            'method': method,
            'url': url,
            'data': data,
        }

        if response is None:
            response_data = None
        else:
            response_data = {
                'status_code': response.status_code,
            }
            try:
                response_data['data'] = response.json()
            except Exception:
                LOGGER.exception('Failed to parse Taxi API response:\n%s', response.text)
                response_data['data'] = None
                response_data['text'] = response.text

        log_data['response'] = response_data

        self._push_client.log(type_='api-request', data=log_data)

    def get_order(self, order_id):
        url = self._urls.get_order_url(order_id=order_id)
        response = self._request(url=url)
        order_details = OrderDetails.from_dict(response.json())
        return order_details

    def make_order(self, src_point, dst_point):
        '''
        Make an order.

        Args:
            src_point: (lon, lat)
            dst_point: (lon, lat)
        '''
        src_point_info = self._get_point_info(src_point)
        dst_point_info = self._get_point_info(dst_point)
        order_id = self._create_order(source=src_point_info, destination=dst_point_info)
        self._confirm_order(order_id=order_id)
        return order_id

    def _get_point_info(self, point):
        response = self._geocoder.geocode_coords(lat=point[1], lon=point[0])
        return _PointInfo.from_geocoder_object(point, response.objects[0])

    def _create_order(self, source, destination):
        url = self._urls.get_order_url()
        payload = self._make_order_request_payload(source, destination)
        response = self._request(method='POST', url=url, data=payload)
        if response is None:
            return None
        data = response.json()
        order_id = data['_id']
        return order_id

    def _make_order_request_payload(self, source, destination):
        payload = {
            "city": source.locality,
            "class": "econom",
            "source": source.to_dict(),
            "destination": destination.to_dict(),
            "phone": "+7977818-09-39",
        }
        return payload

    def _confirm_order(self, order_id):
        url = self._urls.get_order_processing_url(order_id=order_id)
        response = self._request(method='POST', url=url)
        return response


class _PointInfo(object):

    def __init__(self, country, locality, fullname, geopoint):
        self.country = country
        self.locality = locality
        self.fullname = fullname
        self.geopoint = geopoint

    @classmethod
    def from_geocoder_object(cls, geopoint, obj):
        metadata = obj.raw['metaDataProperty']['GeocoderMetaData']
        fullname = metadata['text']

        country = None
        locality = None
        for component in metadata['Address']['Components']:
            kind = component['kind']
            name = component['name']
            if kind == 'country':
                country = name
            elif kind == 'locality':
                locality = name

        return cls(
            country=country,
            locality=locality,
            fullname=fullname,
            geopoint=geopoint,
        )


    def to_dict(self):
        return {
            'country': self.country,
            'fullname': self.fullname,
            'geopoint': self.geopoint,
        }

    @classmethod
    def from_dict(cls, data):
        geopoint = data.get('geopoint')
        if geopoint is not None:
            geopoint = [float(coord) for coord in geopoint]
        return cls(
            country=None,
            locality=None,
            fullname=data.get('fullname'),
            geopoint=geopoint,
        )


class OrderDetails(object):

    def __init__(self, id_, source, destination, status):
        self.id = id_
        self.source = source
        self.destination = destination
        self.status = status


    @classmethod
    def from_dict(cls, data):
        id_ = data['_id']
        source = _PointInfo.from_dict(data['source'])
        destination = _PointInfo.from_dict(data['destination'])
        status = OrderStatus.from_dict(data['status'])
        return cls(
            id_=id_,
            source=source,
            destination=destination,
            status=status,
        )


class OrderStatus(object):

    def __init__(self, description, full, simple):
        self.description = description
        self.full = full
        self.simple = simple

    @classmethod
    def from_dict(cls, data):
        return cls(
            description=data.get('description'),
            full=data.get('full'),
            simple=data.get('simple'),
        )
