import concurrent.futures
import logging
import queue
import threading
import time

from ...solomon import SOLOMON


LOGGER = logging.getLogger(__name__)


class BasePacketHandler(object):

    @classmethod
    def get_handler_id(cls):
        raise NotImplementedError

    def handle_packet(self, imei, packet):
        raise NotImplementedError


class BaseThreadedPacketHandler(BasePacketHandler):

    def __init__(self, *args, max_workers=4, **kwargs):
        super().__init__(*args, **kwargs)
        self._queue = queue.Queue(maxsize=16384)
        self._handler_thread = self._spawn_handler_thread()
        self._monitoring_thread = self._spawn_monitoring_thread()
        self._pool = concurrent.futures.ThreadPoolExecutor(max_workers=max_workers)

    def _handle_packet(self, imei, packet):
        raise NotImplementedError

    def handle_packet(self, imei, packet):
        self._queue.put((imei, packet))

    def _spawn_handler_thread(self):
        thread = threading.Thread(target=self._run_handler, daemon=True)
        thread.start()
        return thread

    def _run_handler(self):
        while True:
            try:
                self._run_handler_iteration()
            except Exception:
                LOGGER.exception('handler failed')

    def _run_handler_iteration(self):
        imei, packet = self._queue.get()
        self._pool.submit(self._handle_packet, imei=imei, packet=packet)

    def _spawn_monitoring_thread(self):
        thread = threading.Thread(target=self._run_monitoring, daemon=True)
        thread.start()
        return thread

    def _run_monitoring(self):
        queue_size_sensor = 'packet_handlers.{}.queue.size'.format(self.get_handler_id())
        while True:
            qsize = self._queue.qsize()
            SOLOMON.set_value(queue_size_sensor, qsize)
            time.sleep(1)
