import json
import logging
import queue
import time
from abc import abstractmethod
from collections import defaultdict
from threading import Lock

import furl
from kubiki.util import make_requests_session

from saaspy import SaasClient, SaasDocument

import cars.settings
from cars.telematics.backend.typed_packets import (
    WialonCombineDataPacket,
    WialonCombineDataPacketCustomParameters,
    WialonCombineDataPacketPositionData
)
from cars.telematics.backend.packet_handlers.base import BaseThreadedPacketHandler


LOGGER = logging.getLogger(__name__)


class BaseCarStateCache:

    def __init__(self, cached_state_ttl, state_check_base_furl):
        self._cached_state_ttl = cached_state_ttl
        self._state_check_base_furl = state_check_base_furl

        self._state_updated_at = {}
        self._last_state = {}
        self._session = make_requests_session(retries=0)

        self._locks = defaultdict(Lock)

    def get_id(self, imei):
        if imei not in self._last_state:
            self._update_state_if_needed(imei)
        return self._last_state[imei]['id']

    def _update_state_if_needed(self, imei):
        time_after_last_update = time.time() - self._state_updated_at.get(imei, 0)
        if time_after_last_update > self._cached_state_ttl:
            self._locks[imei].acquire()
            try:
                LOGGER.info(
                    'updating data for car with imei=%s in cache %s',
                    str(imei),
                    self.__class__.__name__,
                )
                r = self._session.get(self._get_state_check_url(imei), timeout=1)
                r.raise_for_status()
                self._last_state[imei] = r.json()
                self._state_updated_at[imei] = time.time()
            except Exception:
                LOGGER.exception(
                    'failed to update cache for car with imei=%s in cache %s',
                    str(imei),
                    self.__class__.__name__,
                )
                self._locks[imei].release()
                raise

            self._locks[imei].release()

    def _get_state_check_url(self, imei):
        state_check_url = self._state_check_base_furl.copy().add({'imei': imei}).url
        return state_check_url


class CarCurrentRideCache(BaseCarStateCache):

    @classmethod
    def from_settings(cls):
        config = cars.settings.TELEMATICS['packet_handlers']['saas_submission']['ride_cache']
        return cls(
            cached_state_ttl=config['cached_state_ttl'],
            state_check_base_furl=furl.furl(config['get_state_url']),
        )

    def get_current_ride_id(self, imei):
        self._update_state_if_needed(imei)
        return self._last_state[imei]['current_ride_id']

    def get_current_user_id(self, imei):
        self._update_state_if_needed(imei)
        return self._last_state[imei]['current_user_id']

    def get_current_order_id(self, imei):
        self._update_state_if_needed(imei)
        return self._last_state[imei]['current_order_id']


class CarStatusCache(BaseCarStateCache):

    @classmethod
    def from_settings(cls):
        config = cars.settings.TELEMATICS['packet_handlers']['saas_submission']['status_cache']
        return cls(
            cached_state_ttl=config['cached_state_ttl'],
            state_check_base_furl=furl.furl(config['get_state_url']),
        )

    def get_status(self, imei):
        self._update_state_if_needed(imei)
        return self._last_state[imei]['status']


class BaseSaasSubmitterPacketHandler(BaseThreadedPacketHandler):

    current_ride_cache = CarCurrentRideCache.from_settings()
    status_cache = CarStatusCache.from_settings()

    def __init__(self, saas_client, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._client = saas_client

    @classmethod
    def create_saas_drive_client(cls):
        general_config = cars.settings.TELEMATICS['packet_handlers']['saas_submission']

        saas_config = general_config['drive_service']
        saas_client = SaasClient(
            index_host=saas_config['index_host'],
            search_host=None,
            service_name=saas_config['service_name'],
            key=saas_config['key'],
            timeout=saas_config['timeout'],
        )

        return saas_client

    def _form_saas_car_riding_document(self, lat, lon, car_id, user_id, ride_id, order_id,
                                       speed, timestamp, course, g_x=None, g_y=None,
                                       g_z=None, rpm=None):
        # According to https://st.yandex-team.ru/DRIVEBACK-17#1515599699000
        # Convention: use IDs from admin portal for car_id (device id), user_id and ride_id

        doc = SaasDocument(ride_id)

        doc.add_search_attr_literal('s_device_id', car_id)  # device_id = car_id
        doc.add_property('s_speed', str(speed))
        doc.add_property('s_course', str(course))
        doc.add_property('ACCID', user_id)
        doc.add_property('ModificationType', 'mtTracePoint')
        doc.add_property('TS', int(timestamp))
        doc.add_property('RHASH', ride_id)
        doc.add_property('DID', car_id)
        doc.add_property('SESSIONID', order_id)
        doc.add_geodata(
            'geo',
            {
                'type': 'DRIVE_RIDING',
                'coordinates': [{'X': lon, 'Y': lat}],
            },
        )

        if g_x is not None:
            doc.add_property('g_x', g_x)
        if g_y is not None:
            doc.add_property('g_y', g_y)
        if g_z is not None:
            doc.add_property('g_z', g_z)
        if rpm is not None:
            doc.add_property('rpm', rpm)

        doc.modification_timestamp = int(timestamp)
        # (0, 0) stands for "no destination is known"
        # we will pass the destination coords, when the fix price scenario is implemented
        # the format in this case should be "lon lat"
        doc.add_property(
            'destination',
            '{} {}'.format(0, 0),
        )

        return doc

    def _handle_packet(self, imei, packet):
        if not isinstance(packet, WialonCombineDataPacket):
            return

        for record in packet.records:
            position_subrecord = None
            custom_parameters_subrecord = None

            for subrecord in record.subrecords:
                if isinstance(subrecord, WialonCombineDataPacketPositionData):
                    position_subrecord = subrecord
                if isinstance(subrecord, WialonCombineDataPacketCustomParameters):
                    custom_parameters_subrecord = subrecord

            if position_subrecord is not None:
                try:
                    self._handle_subrecord(
                        imei=imei,
                        position_subrecord=position_subrecord,
                        custom_parameters_subrecord=custom_parameters_subrecord,
                        timestamp=record.timestamp
                    )
                except Exception:
                    LOGGER.exception('Error while handling subrecord!')

    def _handle_saas_car_riding_subrecord(self, imei, position_subrecord,
                                          custom_parameters_subrecord, timestamp):
        current_ride_id = self.current_ride_cache.get_current_ride_id(imei)
        if not current_ride_id:
            # Not in a ride now
            return
        car_id = self.current_ride_cache.get_id(imei)
        current_user_id = self.current_ride_cache.get_current_user_id(imei)
        current_order_id = self.current_ride_cache.get_current_order_id(imei)

        document_attrs = {
            'lat': position_subrecord.lat,
            'lon': position_subrecord.lon,
            'car_id': car_id,
            'user_id': current_user_id,
            'ride_id': current_ride_id,
            'order_id': current_order_id,
            'speed': position_subrecord.speed,
            'timestamp': timestamp,
            'course': position_subrecord.course,
        }

        if custom_parameters_subrecord is not None:
            document_attrs.update(
                {
                    'g_x': custom_parameters_subrecord.params.get(1243),
                    'g_y': custom_parameters_subrecord.params.get(1244),
                    'g_z': custom_parameters_subrecord.params.get(1245),
                    'rpm': custom_parameters_subrecord.params.get(2109),
                }
            )

        saas_document = self._form_saas_car_riding_document(**document_attrs)

        self._client.modify_document(saas_document, realtime=True)

    @abstractmethod
    def _handle_subrecord(self, imei, position_subrecord, custom_parameters_subrecord, timestamp):
        raise NotImplementedError


class CarStateCacheStub:

    def __init__(self):
        self.documents = queue.Queue()

    def get_id(self, imei):  # pylint: disable=unused-argument
        return 'stub-car-guid'

    def get_current_ride_id(self, imei):  # pylint: disable=unused-argument
        return 'stub-ride-guid'

    def get_current_user_id(self, imei):  # pylint: disable=unused-argument
        return 'stub-user-guid'

    def get_status(self, imei):  # pylint: disable=unused-argument
        return 'ride'

    def get_current_order_id(self, imei):  # pylint: disable=unused-argument
        return 'stub-order-id'


class SaasClientStub(SaasClient):

    def __init__(self):
        super().__init__(None, None, None, None)
        self.documents = queue.Queue()

    def modify_document(self, document, realtime=False, prefix=None, extra_params=None):
        LOGGER.info('Adding document: %s', str(document.to_json()))
        self.documents.put(document)
