import time
import threading
from collections import defaultdict

from cars.telematics.backend.typed_packets import WialonCombineDataPacketPositionData

import cars.settings

from kubiki.util import make_requests_session

from .base import BaseThreadedPacketHandler


class SignalCollectorPacketHandler(BaseThreadedPacketHandler):
    """Send data packets to collector API"""

    def __init__(self, *args, url, timeout, **kwargs):
        n_workers = kwargs.setdefault('max_workers', 32)
        super().__init__(*args, **kwargs)

        self._url = url
        self._timeout = timeout

        self._session = make_requests_session(pool_connections=n_workers, pool_maxsize=n_workers)
        self._last_submission = defaultdict(int)
        self._lock = threading.Lock()

    @classmethod
    def get_handler_id(cls):
        return 'signal_collector'

    @classmethod
    def from_settings(cls):
        settings = cars.settings.TELEMATICS['packet_handlers']['signal_collector']
        return cls(
            url=settings['url'],
            timeout=settings['timeout'],
        )

    def _handle_packet(self, imei, packet):
#        print('\n\n\nI ENTER\n\n\n')
        packet_type = packet.get_type()
        if imei is None or packet_type != 'data':
#            print('\n\n\nNO DATA I RETURN\n\n\n')
            return

        data = {
            'imei': imei,
            'packet': packet.to_dict(),
        }

        has_position = False
        for record in packet.records:
            for subrecord in record.subrecords:
                if isinstance(subrecord, WialonCombineDataPacketPositionData):
                    has_position = True
                    break

        if not has_position:
#            print('\n\n\n\nNO POSITION I RETURN\n\n\n')
            return

        cur_time = time.time()
        self._lock.acquire()
        if self._last_submission[imei] + 10 <= cur_time:
#            print('\n\n\nI SUBMIT\n\n\n')
            self._last_submission[imei] = cur_time
            self._lock.release()
            response = self._session.post(self._url, json=data, timeout=self._timeout)
            response.raise_for_status()
        else:
 #           print('\n\n\nTOO OLD I RETURN\n\n\n')
            pass
