import collections
import logging
import struct
import uuid

from twisted.internet.protocol import Factory, Protocol

from .generic_packet import ChecksumMismatchError, WialonCombineGenericPacket
from .typed_packets import WialonCombinePacketFactory


LOGGER = logging.getLogger(__name__)


class WialonCombineProtocol(Protocol):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._id = uuid.uuid1().hex
        self._buffer = collections.deque()

    @property
    def id(self):
        return self._id

#    def drop(self):
#        self._log(logging.WARNING, "Dropping connection")
#        self.transport.loseConnection()

    def get_server(self):
        return self.factory.server  # pylint: disable=no-member

    def get_imei(self):
        return self.get_server().get_imei(self)

    def send(self, data):
        self.transport.write(data)

    def send_command(self, command):
        parts = []

        if command.id:
            parts.extend([b'/', command.id.encode()])

        parts.extend([b'#', command.text.encode(), b'#\r\n'])

        data = b''.join(parts)

#        self._log(logging.INFO, 'sending %s', repr(data))

        return self.send(data)

    def connectionLost(self, *args, **kwargs):
        self._log(logging.INFO, 'Connection lost')
        self.get_server().handle_connection_lost(self)
        return super().connectionLost(*args, **kwargs)

    def connectionMade(self, *args, **kwargs):
        self._log(logging.INFO, 'Connection made')
        return super().connectionMade(*args, **kwargs)

    def dataReceived(self, data):
        self._log(logging.INFO, 'Data received: %s', repr(data))
        self._buffer.append(data)

        while True:
            generic_packet = self._try_parse_generic_packet()
            if generic_packet is None:
                break

            packet = WialonCombinePacketFactory.from_generic_packet(generic_packet)
            try:
                self.get_server().handle_packet(self, packet)
            except Exception:
                self._exception('Failed to handle a packet: %s', packet)
            if self.get_imei() is not None:
                self._reply(generic_packet)

#    def logPrefix(self, data):
#        return self._id
#
    def _try_parse_generic_packet(self):
        try:
            packet = WialonCombineGenericPacket.try_parse_from_buffer(self._buffer)
        except ChecksumMismatchError:
            self._exception('Packet checksum mismatch')
            packet = None
        except Exception:
            self._exception(
                'Failed to parse buffer:\n%s',
                repr(b''.join(self._buffer)),
            )
            if self._buffer:
                # Remove the piece that may have caused the exception.
                self._buffer.popleft()
            packet = None
        return packet

    def _parse_generic_packet(self, generic_packet):
        pass

    def _reply(self, packet):
        seq_bytes = struct.pack('>H', packet.seq)
        data = b'@@\x00' + seq_bytes
        self._log(logging.INFO, 'Replying: %s', repr(data))
        self.transport.write(data)

    def _log(self, level, msg, *args, **kwargs):
#        msg = 'id={}\timei={}\t{}'.format(self._id, self.get_imei(), msg)
        msg = 'imei={}\t{}'.format(self.get_imei(), msg)
        LOGGER.log(level, msg, *args, **kwargs)

    def _exception(self, msg, *args, **kwargs):
#        msg = 'id={}\timei={}\t{}'.format(self._id, self.get_imei(), msg)
        msg = 'imei={}\t{}'.format(self.get_imei(), msg)
        LOGGER.exception(msg, *args, **kwargs)


class WialonCombineFactory(Factory):

    protocol = WialonCombineProtocol

    def __init__(self, server):
        self.server = server
