import logging

from ..solomon import SOLOMON


LOGGER = logging.getLogger(__name__)


class ProtocolRegistry(object):

    class NotFound(Exception):
        pass

    def __init__(self):
        self._imei2protocol = {}
        self._protocol2imei = {}

    def get_imei(self, protocol):
        try:
            return self._protocol2imei[protocol.id]
        except KeyError as e:
            raise self.NotFound from e

    def get_protocol(self, imei):
        try:
            return self._imei2protocol[imei]
        except KeyError as e:
            raise self.NotFound from e

    def register(self, imei, protocol):
        self._monitor()

        if imei in self._imei2protocol:
            LOGGER.error('Registry already contains imei %s', imei)

        self._imei2protocol[imei] = protocol
        self._protocol2imei[protocol.id] = imei

        LOGGER.info('Registered imei %s for protocol %s', imei, protocol.id)

    def unregister(self, imei):
        self._monitor()

        try:
            protocol = self._imei2protocol.pop(imei)
        except KeyError as e:
            LOGGER.exception('Attempting to unregisterer imei %s which was not registered', imei)
            raise self.NotFound from e

        self._protocol2imei.pop(protocol.id)

        LOGGER.info('Unregistered imei %s for protocol %s', imei, protocol.id)

    def _monitor(self):
        SOLOMON.set_value('protocol_registry.imei2protocol.size', len(self._imei2protocol))
        SOLOMON.set_value('protocol_registry.protocol2imei.size', len(self._protocol2imei))
