import logging
import re

from ..commands.registry import CommandRegistry
from .protocol import WialonCombineFactory
from .protocol_registry import ProtocolRegistry
from .typed_packets import (
    WialonCombineLoginPacket, WialonCombineDataPacket, WialonCombineDataPacketDriverMessage,
)


LOGGER = logging.getLogger(__name__)


class TelematicsBackendServer(object):

    def get_factory(self):
        raise NotImplementedError

    def handle_connection_lost(self):
        raise NotImplementedError

    def handle_login(self, protocol, packet):
        raise NotImplementedError

    def handle_packet(self, protocol, packet):
        raise NotImplementedError

    def send_command(self, command):
        raise NotImplementedError


class WialonCombineTelematicsBackendServer(TelematicsBackendServer):

    def __init__(self, packet_handlers, push_client):
        self._factory = WialonCombineFactory(server=self)
        self._protocol_registry = ProtocolRegistry()
        self._push_client = push_client
        self._command_registry = CommandRegistry()
        self._packet_handlers = packet_handlers

    def get_factory(self):
        return self._factory

    def get_imei(self, protocol):
        try:
            imei = self._protocol_registry.get_imei(protocol)
        except self._protocol_registry.NotFound:
            imei = None
        return imei

    def handle_connection_lost(self, protocol):
        imei = self.get_imei(protocol)
        if imei is not None:
            try:
                self._protocol_registry.unregister(imei)
            except self._protocol_registry.NotFound:
                pass

    def handle_packet(self, protocol, packet):
        if isinstance(packet, WialonCombineLoginPacket):
            self.handle_login(protocol, packet)
        elif isinstance(packet, WialonCombineDataPacket):
            self.handle_data(packet)

        imei = self.get_imei(protocol)
        for handler in self._packet_handlers:
            try:
                handler.handle_packet(imei, packet)
            except Exception:
                LOGGER.exception('packet handler failed: %s', handler)

    def handle_login(self, protocol, packet):
        self._protocol_registry.register(imei=packet.id, protocol=protocol)

    def handle_data(self, packet):
        for record in packet.records:
            for subrecord in record.subrecords:
                if isinstance(subrecord, WialonCombineDataPacketDriverMessage):
                    self.handle_data_driver_message(subrecord.text)

    def handle_data_driver_message(self, text):
        match = re.match(r'^/(?P<command_id>\w+) (?P<text>.+)\n$', text, re.IGNORECASE)
        if match is None:
            LOGGER.warning('unparsed driver message: %s', repr(text))
            return

        command_id = match.group('command_id')
        command_text = match.group('text')

        command = self._command_registry.get(command_id)
        if command is None:
            LOGGER.warning('received a reply for an unknown command: %s', repr(text))
            return

        try:
            command.handle_reply(command_text)
        except Exception:
            LOGGER.exception('failed to handle command reply: %s', repr(text))
            return

    def send_command(self, command):
        protocol = self._protocol_registry.get_protocol(imei=command.imei)

        if command.id:
            self._command_registry.register(command=command)

        protocol.send_command(command)

        command.deferred.addCallback(
            lambda result: self._on_command_result(command, result),
        )

    def _on_command_result(self, command, result):
        self._push_client.log(imei=command.imei, type_='command', data=command.to_dict())
        return result


class DummyTelematicsBackendServer(TelematicsBackendServer):

    def get_factory(self):
        return None

    def get_imei(self, protocol):  # pylint: disable=unused-argument
        return 0

    def handle_connection_lost(self, protocol):
        pass

    def handle_packet(self, protocol, packet):
        pass

    def handle_login(self, protocol, packet):
        pass

    def send_command(self, command):
        pass
